{{-- File: resources/views/guests/show.blade.php --}}

@extends('admin-layouts.master')

@section('title', 'Detail Tamu - ' . $guest->name)

<style>
    :root {
        --primary-gradient: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        --success-gradient: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
        --info-gradient: linear-gradient(135deg, #3f51b5 0%, #2196f3 100%);
        --warning-gradient: linear-gradient(135deg, #ff9800 0%, #ffc107 100%);
        --danger-gradient: linear-gradient(135deg, #e91e63 0%, #f44336 100%);
    }

    .profile-hero {
       
        border-radius: 20px;
        position: relative;
        overflow: hidden;
        margin-bottom: 2rem;
    }

    .profile-hero::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grain" width="100" height="100" patternUnits="userSpaceOnUse"><circle cx="25" cy="25" r="1" fill="white" opacity="0.1"/><circle cx="75" cy="75" r="1" fill="white" opacity="0.1"/><circle cx="50" cy="10" r="0.5" fill="white" opacity="0.1"/></pattern></defs><rect width="100" height="100" fill="url(%23grain)"/></svg>');
        opacity: 0.3;
    }

    .profile-avatar {
        width: 120px;
        height: 120px;
        border-radius: 50%;
        background: white;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 3rem;
        font-weight: bold;
        color: #667eea;
        box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        position: relative;
        z-index: 2;
    }

    .stats-card {
        background: white;
        border-radius: 15px;
        padding: 1.5rem;
        text-align: center;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        border: none;
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }

    .stats-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 4px;
        background: var(--primary-gradient);
    }

    .stats-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 15px 35px rgba(0,0,0,0.15);
    }

    .stats-number {
        font-size: 2.5rem;
        font-weight: bold;
        margin-bottom: 0.5rem;
        background: var(--primary-gradient);
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        background-clip: text;
    }

    .info-card {
        background: white;
        border-radius: 15px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        border: none;
        overflow: hidden;
        margin-bottom: 1.5rem;
    }

    .info-card-header {
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        padding: 1rem 1.5rem;
        border-bottom: 1px solid #dee2e6;
        position: relative;
    }

    .info-card-header::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 3px;
        background: var(--primary-gradient);
    }

    .info-card-body {
        padding: 1.5rem;
    }

    .info-item {
        display: flex;
        align-items: flex-start;
        margin-bottom: 1rem;
        padding: 0.75rem;
        background: #f8f9fa;
        border-radius: 10px;
        transition: all 0.2s ease;
    }

    .info-item:hover {
        background: #e3f2fd;
        transform: translateX(5px);
    }

    .info-item:last-child {
        margin-bottom: 0;
    }

    .info-icon {
        width: 40px;
        height: 40px;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-right: 1rem;
        color: white;
        font-size: 1.1rem;
    }

    .info-icon.primary { background: var(--primary-gradient); }
    .info-icon.success { background: var(--success-gradient); }
    .info-icon.info { background: var(--info-gradient); }
    .info-icon.warning { background: var(--warning-gradient); }
    .info-icon.danger { background: var(--danger-gradient); }

    .booking-card {
        background: white;
        border-radius: 12px;
        margin-bottom: 1rem;
        overflow: hidden;
        transition: all 0.3s ease;
        border: 1px solid #e9ecef;
    }

    .booking-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 25px rgba(0,0,0,0.12);
    }

    .booking-header {
        padding: 1rem 1.5rem;
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        border-bottom: 1px solid #dee2e6;
    }

    .status-badge {
        padding: 0.5rem 1rem;
        border-radius: 20px;
        font-size: 0.85rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .status-booked { 
        background: linear-gradient(135deg, #3f51b5, #2196f3); 
        color: white; 
    }
    
    .status-checked_in { 
        background: linear-gradient(135deg, #ff9800, #ffc107); 
        color: white; 
    }
    
    .status-checked_out { 
        background: linear-gradient(135deg, #11998e, #38ef7d); 
        color: white; 
    }
    
    .status-cancelled { 
        background: linear-gradient(135deg, #e91e63, #f44336); 
        color: white; 
    }

    .timeline {
        position: relative;
        padding-left: 2rem;
    }

    .timeline::before {
        content: '';
        position: absolute;
        left: 1rem;
        top: 0;
        bottom: 0;
        width: 2px;
        background: linear-gradient(to bottom, #667eea, #764ba2);
    }

    .timeline-item {
        position: relative;
        margin-bottom: 2rem;
        background: white;
        border-radius: 10px;
        padding: 1rem;
        box-shadow: 0 2px 10px rgba(0,0,0,0.08);
    }

    .timeline-item::before {
        content: '';
        position: absolute;
        left: -1.75rem;
        top: 1rem;
        width: 12px;
        height: 12px;
        border-radius: 50%;
        background: #667eea;
        border: 3px solid white;
        box-shadow: 0 0 0 3px #667eea;
    }

    .action-buttons {
        position: sticky;
        top: 20px;
        z-index: 10;
    }

    .btn-gradient {
        background: var(--primary-gradient);
        border: none;
        color: white;
        padding: 0.75rem 1.5rem;
        border-radius: 10px;
        font-weight: 600;
        transition: all 0.3s ease;
        text-decoration: none;
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
    }

    .btn-gradient:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 20px rgba(102, 126, 234, 0.3);
        color: white;
        text-decoration: none;
    }

    .btn-success-gradient {
        background: var(--success-gradient);
    }

    .btn-success-gradient:hover {
        box-shadow: 0 8px 20px rgba(17, 153, 142, 0.3);
    }

    .btn-warning-gradient {
        background: var(--warning-gradient);
    }

    .btn-warning-gradient:hover {
        box-shadow: 0 8px 20px rgba(255, 152, 0, 0.3);
    }

    .btn-danger-gradient {
        background: var(--danger-gradient);
    }

    .btn-danger-gradient:hover {
        box-shadow: 0 8px 20px rgba(233, 30, 99, 0.3);
    }

    .quick-stats {
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        border-radius: 15px;
        padding: 1.5rem;
        margin-bottom: 1.5rem;
    }

    .stat-item {
        text-align: center;
        padding: 1rem;
    }

    .stat-value {
        font-size: 1.8rem;
        font-weight: bold;
        color: #667eea;
        margin-bottom: 0.25rem;
    }

    .stat-label {
        color: #6c757d;
        font-size: 0.9rem;
        font-weight: 500;
    }

    .empty-state {
        text-align: center;
        padding: 3rem 1rem;
        color: #6c757d;
    }

    .empty-state i {
        font-size: 4rem;
        color: #dee2e6;
        margin-bottom: 1rem;
    }

    .breadcrumb {
        background: none;
        padding: 0;
        margin-bottom: 1rem;
    }

    .breadcrumb-item + .breadcrumb-item::before {
        content: ">";
        color: #6c757d;
    }

    @media (max-width: 768px) {
        .profile-avatar {
            width: 100px;
            height: 100px;
            font-size: 2.5rem;
        }

        .stats-number {
            font-size: 2rem;
        }

        .action-buttons {
            position: static;
            margin-bottom: 1rem;
        }

        .btn-gradient {
            width: 100%;
            margin-bottom: 0.5rem;
            justify-content: center;
        }
    }
</style>

@section('content')
<div class="container-fluid">
    <!-- Breadcrumb -->
    <nav aria-label="breadcrumb" class="mb-4">
        <ol class="breadcrumb">
            <li class="breadcrumb-item"><a href="{{ route('guests.index') }}">Tamu</a></li>
            <li class="breadcrumb-item active" aria-current="page">{{ $guest->name }}</li>
        </ol>
    </nav>

    <!-- Profile Hero Section -->
    <div class="profile-hero bg-primary">
        <div class="container-fluid">
            <div class="row align-items-center" style="min-height: 200px; position: relative; z-index: 2;">
                <div class="col-lg-2 col-md-3 text-center">
                    <div class="profile-avatar mx-auto">
                        {{ strtoupper(substr($guest->name, 0, 1)) }}
                    </div>
                </div>
                <div class="col-lg-7 col-md-6">
                    <div class="text-white">
                        <h1 class="h2 mb-2 font-weight-bold">{{ $guest->name }}</h1>
                        <div class="d-flex flex-wrap gap-3 mb-3">
                            @if($guest->phone)
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-phone mr-2"></i>
                                    <span>{{ $guest->phone }}</span>
                                </div>
                            @endif
                            @if($guest->email)
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-envelope mr-2"></i>
                                    <span>{{ $guest->email }}</span>
                                </div>
                            @endif
                        </div>
                        <div class="d-flex flex-wrap align-items-center gap-2">
                            <span class="badge badge-light px-3 py-2">
                                <i class="fas fa-calendar mr-1"></i>
                                Terdaftar {{ $guest->created_at->diffForHumans() }}
                            </span>
                            @if($guest->bookings->count() >= 5)
                                <span class="badge badge-warning px-3 py-2">
                                    <i class="fas fa-crown mr-1"></i>
                                    VIP Guest
                                </span>
                            @endif
                            @php
                                $hasActiveBooking = $guest->bookings->whereIn('status', ['booked', 'checked_in'])->count() > 0;
                            @endphp
                            @if($hasActiveBooking)
                                <span class="badge badge-success px-3 py-2">
                                    <i class="fas fa-bed mr-1"></i>
                                    Sedang Menginap
                                </span>
                            @endif
                        </div>
                    </div>
                </div>
                <div class="col-lg-3 col-md-3">
                    <div class="action-buttons">
                        <a href="{{ route('guests.edit', $guest->id) }}" class="btn btn-warning">
                            <i class="fas fa-edit"></i>
                            Edit Data
                        </a>
                        {{-- <button class="btn btn-secondary mb-2" onclick="createBooking()">
                            <i class="fas fa-plus-circle"></i>
                            Buat Booking
                        </button> --}}
                       
                        <button class="btn btn-danger" onclick="deleteGuest()">
                            <i class="fas fa-trash"></i>
                            Hapus
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Quick Stats -->
    <div class="quick-stats">
        <div class="row">
            <div class="col-md-3">
                <div class="stat-item">
                    <div class="stat-value">{{ $guest->bookings->count() }}</div>
                    <div class="stat-label">Total Booking</div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-item">
                    <div class="stat-value">{{ $guest->bookings->where('status', 'checked_out')->count() }}</div>
                    <div class="stat-label">Selesai Menginap</div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-item">
                    <div class="stat-value">
                        @php
                            $totalSpent = $guest->bookings->sum(function($booking) {
                                return $booking->bookingDetails->sum('grand_total');
                            });
                        @endphp
                        Rp {{ number_format($totalSpent, 0, ',', '.') }}
                    </div>
                    <div class="stat-label">Total Pengeluaran</div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-item">
                    <div class="stat-value">
                        @php
                            $lastBooking = $guest->bookings->sortByDesc('created_at')->first();
                        @endphp
                        {{ $lastBooking ? $lastBooking->created_at->diffForHumans() : 'Belum ada' }}
                    </div>
                    <div class="stat-label">Booking Terakhir</div>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <!-- Guest Information -->
        <div class="col-lg-4">
            <!-- Personal Information -->
            <div class="info-card">
                <div class="info-card-header">
                    <h6 class="mb-0 font-weight-bold text-dark">
                        <i class="fas fa-user mr-2"></i>
                        Informasi Personal
                    </h6>
                </div>
                <div class="info-card-body">
                    <div class="info-item">
                        <div class="info-icon primary">
                            <i class="fas fa-user"></i>
                        </div>
                        <div class="flex-grow-1">
                            <div class="font-weight-bold">Nama Lengkap</div>
                            <div class="text-muted">{{ $guest->name }}</div>
                        </div>
                    </div>

                    @if($guest->phone)
                        <div class="info-item">
                            <div class="info-icon success">
                                <i class="fas fa-phone"></i>
                            </div>
                            <div class="flex-grow-1">
                                <div class="font-weight-bold">Nomor Telepon</div>
                                <div class="text-muted">{{ $guest->phone }}</div>
                            </div>
                            <div>
                                <a href="tel:{{ $guest->phone }}" class="btn btn-sm btn-outline-success">
                                    <i class="fas fa-phone-alt"></i>
                                </a>
                            </div>
                        </div>
                    @endif

                    @if($guest->email)
                        <div class="info-item">
                            <div class="info-icon info">
                                <i class="fas fa-envelope"></i>
                            </div>
                            <div class="flex-grow-1">
                                <div class="font-weight-bold">Email</div>
                                <div class="text-muted">{{ $guest->email }}</div>
                            </div>
                            <div>
                                <a href="mailto:{{ $guest->email }}" class="btn btn-sm btn-outline-info">
                                    <i class="fas fa-envelope"></i>
                                </a>
                            </div>
                        </div>
                    @endif

                    @if($guest->identity_type && $guest->identity_number)
                        <div class="info-item">
                            <div class="info-icon warning">
                                <i class="fas fa-id-card"></i>
                            </div>
                            <div class="flex-grow-1">
                                <div class="font-weight-bold">{{ $guest->identity_type ?? 'Identitas' }}</div>
                                <div class="text-muted">{{ $guest->identity_number }}</div>
                            </div>
                        </div>
                    @endif

                    @if($guest->address)
                        <div class="info-item">
                            <div class="info-icon danger">
                                <i class="fas fa-map-marker-alt"></i>
                            </div>
                            <div class="flex-grow-1">
                                <div class="font-weight-bold">Alamat</div>
                                <div class="text-muted">{{ $guest->address }}</div>
                            </div>
                        </div>
                    @endif

                    <div class="info-item">
                        <div class="info-icon primary">
                            <i class="fas fa-calendar-plus"></i>
                        </div>
                        <div class="flex-grow-1">
                            <div class="font-weight-bold">Tanggal Registrasi</div>
                            <div class="text-muted">
                                {{ $guest->created_at->format('d F Y, H:i') }}
                                <small class="d-block">{{ $guest->created_at->diffForHumans() }}</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Guest Statistics -->
            <div class="info-card">
                <div class="info-card-header">
                    <h6 class="mb-0 font-weight-bold text-dark">
                        <i class="fas fa-chart-bar mr-2"></i>
                        Statistik Tamu
                    </h6>
                </div>
                <div class="info-card-body">
                    <div class="row text-center">
                        <div class="col-6">
                            <div class="stats-card">
                                <div class="stats-number">{{ $guest->bookings->count() }}</div>
                                <div class="text-muted small">Total Booking</div>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="stats-card">
                                <div class="stats-number">{{ $guest->bookings->where('status', 'checked_out')->count() }}</div>
                                <div class="text-muted small">Selesai</div>
                            </div>
                        </div>
                        <div class="col-6 mt-3">
                            <div class="stats-card">
                                <div class="stats-number">{{ $guest->bookings->where('status', 'cancelled')->count() }}</div>
                                <div class="text-muted small">Dibatalkan</div>
                            </div>
                        </div>
                        <div class="col-6 mt-3">
                            <div class="stats-card">
                                <div class="stats-number">
                                    @php
                                        $totalNights = $guest->bookings->sum(function($booking) {
                                            return $booking->check_in_date && $booking->check_out_date 
                                                ? $booking->check_in_date->diffInDays($booking->check_out_date) 
                                                : 0;
                                        });
                                    @endphp
                                    {{ $totalNights }}
                                </div>
                                <div class="text-muted small">Total Malam</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Booking History -->
        <div class="col-lg-8">
            <div class="info-card">
                <div class="info-card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <h6 class="mb-0 font-weight-bold text-dark">
                            <i class="fas fa-history mr-2"></i>
                            Riwayat Booking ({{ $guest->bookings->count() }})
                        </h6>
                        <button class="btn btn-sm btn-outline-primary" onclick="refreshBookings()">
                            <i class="fas fa-sync-alt mr-1"></i>
                            Refresh
                        </button>
                    </div>
                </div>
                <div class="info-card-body">
                    @if($guest->bookings->count() > 0)
                        <div class="timeline">
                            @foreach($guest->bookings->sortByDesc('created_at') as $booking)
                                <div class="timeline-item">
                                    <div class="booking-card border-0 mb-0">
                                        <div class="booking-header">
                                            <div class="d-flex justify-content-between align-items-start">
                                                <div>
                                                    <h6 class="mb-1 font-weight-bold">
                                                        Booking #{{ $booking->id }}
                                                    </h6>
                                                    <small class="text-muted">
                                                        <i class="fas fa-calendar mr-1"></i>
                                                        Dibuat {{ $booking->created_at->format('d F Y, H:i') }}
                                                    </small>
                                                </div>
                                                <span class="status-badge status-{{ $booking->status }}">
                                                    @switch($booking->status)
                                                        @case('booked')
                                                            <i class="fas fa-calendar-check mr-1"></i>Booked
                                                            @break
                                                        @case('checked_in')
                                                            <i class="fas fa-sign-in-alt mr-1"></i>Check-in
                                                            @break
                                                        @case('checked_out')
                                                            <i class="fas fa-sign-out-alt mr-1"></i>Check-out
                                                            @break
                                                        @case('cancelled')
                                                            <i class="fas fa-times mr-1"></i>Cancelled
                                                            @break
                                                    @endswitch
                                                </span>
                                            </div>
                                        </div>
                                        
                                        <div class="card-body">
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <strong>Periode Menginap:</strong>
                                                        <div class="text-muted">
                                                            @if($booking->check_in_date && $booking->check_out_date)
                                                                <i class="fas fa-calendar mr-1"></i>
                                                                {{ $booking->check_in_date->format('d M Y') }} - 
                                                                {{ $booking->check_out_date->format('d M Y') }}
                                                                <small class="d-block">
                                                                    ({{ $booking->check_in_date->diffInDays($booking->check_out_date) }} malam)
                                                                </small>
                                                            @else
                                                                <span class="text-muted">Tanggal belum ditentukan</span>
                                                            @endif
                                                        </div>
                                                    </div>
                                                </div>
                                                
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <strong>Kamar:</strong>
                                                        <div class="text-muted">
                                                            @if($booking->bookingDetails && $booking->bookingDetails->count() > 0)
                                                                @foreach($booking->bookingDetails as $detail)
                                                                    <div>
                                                                        <i class="fas fa-bed mr-1"></i>
                                                                        {{ $detail->room->room_number ?? 'N/A' }}
                                                                        @if($detail->room && $detail->room->typeRoom)
                                                                            ({{ $detail->room->typeRoom->name }})
                                                                        @endif
                                                                    </div>
                                                                @endforeach
                                                            @else
                                                                <span class="text-muted">Kamar belum ditentukan</span>
                                                            @endif
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            @if($booking->bookingDetails && $booking->bookingDetails->count() > 0)
                                                <div class="mb-3">
                                                    <strong>Total Biaya:</strong>
                                                    <div class="text-success h5 mb-0">
                                                        Rp {{ number_format($booking->bookingDetails->sum('grand_total'), 0, ',', '.') }}
                                                    </div>
                                                </div>
                                            @endif

                                            @if($booking->cancel_description)
                                                <div class="alert alert-danger mb-0">
                                                    <strong>Alasan Pembatalan:</strong><br>
                                                    {{ $booking->cancel_description }}
                                                </div>
                                            @endif

                                            <div class="d-flex justify-content-end mt-3">
                                                <a href="{{ route('bookings.show', $booking->id) }}" 
                                                   class="btn btn-sm btn-outline-primary mr-2">
                                                    <i class="fas fa-eye mr-1"></i>
                                                    Detail
                                                </a>
                                                @if($booking->status === 'booked')
                                                    <button class="btn btn-sm btn-outline-warning mr-2" 
                                                            onclick="checkinBooking({{ $booking->id }})">
                                                        <i class="fas fa-sign-in-alt mr-1"></i>
                                                        Check-in
                                                    </button>
                                                @endif
                                                @if($booking->status === 'checked_in')
                                                    <button class="btn btn-sm btn-outline-success" 
                                                            onclick="checkoutBooking({{ $booking->id }})">
                                                        <i class="fas fa-sign-out-alt mr-1"></i>
                                                        Check-out
                                                    </button>
                                                @endif
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            @endforeach
                        </div>
                    @else
                        <div class="empty-state">
                            <i class="fas fa-calendar-times"></i>
                            <h5>Belum Ada Booking</h5>
                            <p class="mb-4">Tamu ini belum pernah melakukan booking. Mulai dengan membuat booking pertama.</p>
                            <button class="btn btn-primary" onclick="createBooking()">
                                <i class="fas fa-plus-circle mr-2"></i>
                                Buat Booking Pertama
                            </button>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Quick Booking Modal -->
<div class="modal fade" id="quickBookingModal" tabindex="-1" role="dialog" aria-labelledby="quickBookingModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header" style="background: var(--primary-gradient); color: white;">
                <h5 class="modal-title" id="quickBookingModalLabel">
                    <i class="fas fa-plus-circle mr-2"></i>
                    Buat Booking untuk {{ $guest->name }}
                </h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form id="quickBookingForm" >
                    @csrf
                    <input type="hidden" name="guest_id" value="{{ $guest->id }}" id="guestId">
                    <input type="hidden" name="guest_name" value="{{ $guest->name }}" id="guestName">

                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="font-weight-bold">
                                    <i class="fas fa-calendar-alt mr-1 text-primary"></i>
                                    Check-in
                                </label>
                                <input type="date" class="form-control" name="check_in_date" id="checkinDate" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label class="font-weight-bold">
                                    <i class="fas fa-calendar-alt mr-1 text-danger"></i>
                                    Check-out
                                </label>
                                <input type="date" class="form-control" name="check_out_date" id="checkoutDate" required>
                            </div>
                        </div>
                    </div>

                    {{-- <div class="form-group">
                        <label class="font-weight-bold">
                            <i class="fas fa-bed mr-1 text-success"></i>
                            Tipe Kamar
                        </label>
                        <select class="form-control" name="room_type_id" id="roomTypeSelect" required>
                            <option value="">Pilih Tipe Kamar...</option>
                        </select>
                    </div> --}}

                    <div class="form-group">
                        <label class="font-weight-bold">
                            <i class="fas fa-door-open mr-1 text-info"></i>
                            Kamar Tersedia
                        </label>
                        <select class="form-control" name="room_id" id="roomSelect" required>
                            <option value="">Pilih kamar setelah memilih tipe dan tanggal...</option>
                        </select>
                    </div>

                    <div class="alert alert-info" id="pricePreview" style="display: none;">
                        <h6 class="font-weight-bold mb-2">
                            <i class="fas fa-calculator mr-1"></i>
                            Estimasi Biaya:
                        </h6>
                        <div id="priceDetails"></div>
                    </div>

                    <div class="form-group">
                        <label class="font-weight-bold">Catatan (Opsional)</label>
                        <textarea class="form-control" name="notes" rows="3" placeholder="Tambahkan catatan khusus untuk booking ini..."></textarea>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">
                    <i class="fas fa-times mr-1"></i>
                    Batal
                </button>
                <button type="button" class="btn btn-success" onclick="submitQuickBooking()" id="submitBookingBtn" disabled>
                    <i class="fas fa-check mr-1"></i>
                    Buat Booking
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Message Modal -->
<div class="modal fade" id="messageModal" tabindex="-1" role="dialog" aria-labelledby="messageModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header" style="background: var(--warning-gradient); color: white;">
                <h5 class="modal-title" id="messageModalLabel">
                    <i class="fas fa-sms mr-2"></i>
                    Kirim Pesan ke {{ $guest->name }}
                </h5>
                <button type="button" class="close text-white" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <form id="messageForm">
                    @csrf
                    <div class="form-group">
                        <label class="font-weight-bold">Template Pesan</label>
                        <select class="form-control" id="messageTemplate">
                            <option value="">Pilih Template...</option>
                            <option value="welcome">Selamat Datang</option>
                            <option value="birthday">Ucapan Ulang Tahun</option>
                            <option value="promo">Penawaran Khusus</option>
                            <option value="feedback">Minta Feedback</option>
                            <option value="reminder">Pengingat</option>
                            <option value="custom">Pesan Custom</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="font-weight-bold">Pesan</label>
                        <textarea class="form-control" id="messageContent" rows="5" required placeholder="Tulis pesan Anda..."></textarea>
                        <small class="form-text text-muted">
                            <span id="charCount">0</span>/160 karakter
                        </small>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="custom-control custom-checkbox">
                                <input type="checkbox" class="custom-control-input" id="sendSMS" {{ $guest->phone ? '' : 'disabled' }}>
                                <label class="custom-control-label" for="sendSMS">
                                    <i class="fas fa-sms mr-1"></i>
                                    Kirim via SMS
                                    @if(!$guest->phone)
                                        <br><small class="text-muted">(No. telepon tidak tersedia)</small>
                                    @endif
                                </label>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="custom-control custom-checkbox">
                                <input type="checkbox" class="custom-control-input" id="sendEmail" {{ $guest->email ? '' : 'disabled' }}>
                                <label class="custom-control-label" for="sendEmail">
                                    <i class="fas fa-envelope mr-1"></i>
                                    Kirim via Email
                                    @if(!$guest->email)
                                        <br><small class="text-muted">(Email tidak tersedia)</small>
                                    @endif
                                </label>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">
                    <i class="fas fa-times mr-1"></i>
                    Batal
                </button>
                <button type="button" class="btn btn-warning" onclick="submitMessage()">
                    <i class="fas fa-paper-plane mr-1"></i>
                    Kirim Pesan
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Form -->
<form action="{{ route('guests.destroy', $guest->id) }}" method="POST" id="deleteForm" style="display: none;">
    @csrf
    @method('DELETE')
</form>

@endsection

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<script>
function createBooking() {
    $('#quickBookingModal').modal('show');
}
function submitQuickBooking() {
    const form = $('#quickBookingForm')[0];
    const formData = new FormData(form);
   
    confirmBooking({
        title: 'Konfirmasi Quick Booking',
        text: 'Apakah Anda yakin ingin menyimpan quick booking ini?',
        url: $(form).attr('action'),
        method: $(form).attr('method') || 'POST',
        data: formData,
        processData: false,
        contentType: false,
        onSuccess: function() {
            $('#quickBookingModal').modal('hide');
            location.reload();
        }
    });
}
function confirmBooking(options) {
    Swal.fire({
        title: options.title || 'Konfirmasi Booking',
        text: options.text || 'Apakah Anda yakin ingin melakukan booking?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#28a745',
        cancelButtonColor: '#d33',
        confirmButtonText: 'Ya, Booking!',
        cancelButtonText: 'Batal'
    }).then((result) => {
        if (result.isConfirmed) {
            // Loading
            Swal.fire({
                title: 'Memproses Booking...',
                html: 'Mohon tunggu sebentar',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });
            selectedGuest = {
                        id: $('#guestId').val(),
                        name: $('#guestName').val(),
                        type: 'existing'
            };
            console.log('Selected Guest:', selectedGuest);
            // Kirim AJAX
            $.ajax({
                url: '{{ route("bookings.store") }}',
                method: 'POST',
                data: {
                    guest_data: selectedGuest,
                    room_id: $('#roomSelect').val(),
                    check_in_date: $('#checkInDate').val(),
                    check_out_date: $('#checkOutDate').val()
                },
                processData: options.processData !== false, // default true
                contentType: options.contentType !== false, // default true
                success: function(response) {
                    if (response.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'Booking Berhasil!',
                            html: `
                                <div class="text-left">
                                    <p><strong>ID Booking:</strong> #${response.data.booking_id}</p>
                                    <p><strong>Tamu:</strong> ${response.data.guest_name}</p>
                                    <p><strong>Kamar:</strong> ${response.data.room_number}</p>
                                    <p><strong>Check-in:</strong> ${new Date(response.data.check_in_date).toLocaleDateString('id-ID')}</p>
                                    <p><strong>Check-out:</strong> ${new Date(response.data.check_out_date).toLocaleDateString('id-ID')}</p>
                                    <p><strong>Total:</strong> Rp ${response.data.total_amount.toLocaleString('id-ID')}</p>
                                </div>
                            `,
                            confirmButtonText: 'Lihat Booking'
                        }).then(() => {
                            if (typeof options.onSuccess === 'function') {
                                options.onSuccess(response);
                            }
                        });
                    }
                },
                error: function(xhr) {
                    let errorMsg = 'Terjadi kesalahan saat memproses booking';

                    if (xhr.responseJSON) {
                        if (xhr.status === 422) {
                            const errors = xhr.responseJSON.errors;
                            let errorList = '';
                            Object.keys(errors).forEach(key => {
                                errorList += `<li>${errors[key][0]}</li>`;
                            });
                            errorMsg = `<ul class="text-left">${errorList}</ul>`;
                        } else {
                            errorMsg = xhr.responseJSON.message || errorMsg;
                        }
                    }

                    Swal.fire({
                        icon: 'error',
                        title: 'Booking Gagal!',
                        html: errorMsg
                    });
                }
            });
        }
    });
}

// function confirmBooking() {
//     Swal.fire({
//         title: 'Konfirmasi Booking',
//         text: 'Apakah Anda yakin ingin melakukan booking?',
//         icon: 'question',
//         showCancelButton: true,
//         confirmButtonColor: '#28a745',
//         cancelButtonColor: '#d33',
//         confirmButtonText: 'Ya, Booking!',
//         cancelButtonText: 'Batal'
//     }).then((result) => {
//         if (result.isConfirmed) {
//             // Show loading
//             Swal.fire({
//                 title: 'Memproses Booking...',
//                 html: 'Mohon tunggu sebentar',
//                 allowOutsideClick: false,
//                 didOpen: () => {
//                     Swal.showLoading();
//                 }
//             });
            
//             // AJAX call to save booking
//             $.ajax({
//                 url: '{{ route("bookings.store") }}',
//                 method: 'POST',
//                 data: {
//                     guest_data: selectedGuest,
//                     room_id: selectedRoom.id,
//                     check_in_date: $('#checkInDate').val(),
//                     check_out_date: $('#checkOutDate').val()
//                 },
//                 success: function(response) {
//                     if (response.success) {
//                         Swal.fire({
//                             icon: 'success',
//                             title: 'Booking Berhasil!',
//                             html: `
//                                 <div class="text-left">
//                                     <p><strong>ID Booking:</strong> #${response.data.booking_id}</p>
//                                     <p><strong>Tamu:</strong> ${response.data.guest_name}</p>
//                                     <p><strong>Kamar:</strong> ${response.data.room_number}</p>
//                                     <p><strong>Check-in:</strong> ${new Date(response.data.check_in_date).toLocaleDateString('id-ID')}</p>
//                                     <p><strong>Check-out:</strong> ${new Date(response.data.check_out_date).toLocaleDateString('id-ID')}</p>
//                                     <p><strong>Total:</strong> Rp ${response.data.total_amount.toLocaleString('id-ID')}</p>
//                                 </div>
//                             `,
//                             confirmButtonText: 'Lihat Booking'
//                         }).then(() => {
//                             // Redirect to booking detail or list
//                             window.location.href = `{{ url('bookings') }}`;
//                         });
//                     }
//                 },
//                 error: function(xhr) {
//                     let errorMsg = 'Terjadi kesalahan saat memproses booking';
                    
//                     if (xhr.responseJSON) {
//                         if (xhr.status === 422) {
//                             // Validation errors
//                             const errors = xhr.responseJSON.errors;
//                             let errorList = '';
//                             Object.keys(errors).forEach(key => {
//                                 errorList += `<li>${errors[key][0]}</li>`;
//                             });
//                             errorMsg = `<ul class="text-left">${errorList}</ul>`;
//                         } else {
//                             errorMsg = xhr.responseJSON.message || errorMsg;
//                         }
//                     }
                    
//                     Swal.fire({
//                         icon: 'error',
//                         title: 'Booking Gagal!',
//                         html: errorMsg
//                     });
//                 }
//             });
//         }
//     });
// }
// Fixed checkRoomAvailability function - tanpa perlu room type filter
function checkRoomAvailability() {
    const checkinDate = $('#checkinDate').val();
    const checkoutDate = $('#checkoutDate').val();
    
    // Hanya perlu tanggal, tidak perlu room type
    if (!checkinDate || !checkoutDate) {
        $('#roomSelect').prop('disabled', true).html('<option value="">Pilih tanggal check-in dan check-out...</option>');
        $('#submitBookingBtn').prop('disabled', true);
        $('#pricePreview').hide();
        return;
    }
    
    // Validasi tanggal
    if (new Date(checkinDate) >= new Date(checkoutDate)) {
        $('#roomSelect').prop('disabled', true).html('<option value="">Tanggal check-out harus setelah check-in</option>');
        $('#submitBookingBtn').prop('disabled', true);
        $('#pricePreview').hide();
        return;
    }
    
    // Show loading state
    $('#roomSelect').html('<option value="">🔄 Mencari kamar tersedia...</option>').prop('disabled', true);
    $('#submitBookingBtn').prop('disabled', true);
    $('#pricePreview').hide();
    
    // Call API untuk cek ketersediaan kamar - tanpa filter room type
    $.ajax({
        url: '{{ route("api.bookings.check-availability") }}',
        method: 'POST',
        data: {
            check_in_date: checkinDate,
            check_out_date: checkoutDate,
            // Hapus room_type_id filter
            _token: $('meta[name="csrf-token"]').attr('content')
        },
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json'
        },
        success: function(response) {
            console.log('Room availability response:', response); // Debug log
            
            if (response.success && response.data && response.data.length > 0) {
                // Tampilkan semua kamar yang tersedia
                const availableRooms = response.data;
                
                let options = '<option value="">Pilih Kamar...</option>';
                availableRooms.forEach(function(room) {
                    // Format harga untuk display
                    const formattedPrice = parseInt(room.price_per_night).toLocaleString('id-ID');
                    
                    options += `<option value="${room.id}" 
                                data-room-number="${room.room_number}"
                                data-type-name="${room.type_name}"
                                data-price="${room.price_per_night}"
                                data-ppn="${room.ppn}"
                                data-capacity-adult="${room.max_capacity_adult}"
                                data-capacity-child="${room.max_capacity_child}"
                                data-description="${room.description || ''}">
                        ${room.type_name} - Kamar ${room.room_number} (Rp ${formattedPrice}/malam)
                    </option>`;
                });
                
                $('#roomSelect').html(options).prop('disabled', false);
                
                // Show success message
                showToast(`Ditemukan ${availableRooms.length} kamar tersedia`, 'success');
                
                // Auto-calculate price if only one room available
                if (availableRooms.length === 1) {
                    $('#roomSelect').val(availableRooms[0].id);
                    calculatePrice();
                }
            } else {
                $('#roomSelect').html('<option value="">Tidak ada kamar tersedia</option>');
                
                // Show informative message
                if (response.total_available === 0) {
                    showToast('Semua kamar telah terisi pada tanggal yang dipilih. Silakan pilih tanggal lain.', 'warning');
                } else {
                    showToast('Tidak ada kamar tersedia pada tanggal tersebut', 'info');
                }
            }
        },
        error: function(xhr) {
            console.error('Error checking room availability:', xhr);
            
            let errorMsg = 'Terjadi kesalahan saat mengecek ketersediaan kamar';
            if (xhr.responseJSON && xhr.responseJSON.message) {
                errorMsg = xhr.responseJSON.message;
            }
            
            $('#roomSelect').html(`<option value="">Error: ${errorMsg}</option>`);
            showToast(errorMsg, 'error');
        }
    });
}

// Updated setupEventListeners - hapus room type change listener
function setupEventListeners() {
    // Date change handlers
    $('#checkinDate').change(function() {
        const checkinDate = new Date($(this).val());
        checkinDate.setDate(checkinDate.getDate() + 1);
        const nextDay = checkinDate.toISOString().split('T')[0];
        
        $('#checkoutDate').attr('min', nextDay);
        
        // Auto-set checkout if empty or invalid
        if (!$('#checkoutDate').val() || $('#checkoutDate').val() <= $(this).val()) {
            $('#checkoutDate').val(nextDay);
        }
        
        // Check room availability whenever dates change
        checkRoomAvailability();
    });
    
    $('#checkoutDate').change(function() {
        checkRoomAvailability();
    });
    
    // Room selection handler
    $('#roomSelect').change(function() {
        calculatePrice();
    });
    
    // Message character counter
    $('#messageContent').on('input', function() {
        const length = $(this).val().length;
        $('#charCount').text(length);
        
        if (length > 160) {
            $('#charCount').addClass('text-danger');
        } else {
            $('#charCount').removeClass('text-danger');
        }
    });
    
    // Message template handler
    $('#messageTemplate').change(function() {
        const template = $(this).val();
        let message = '';
        
        switch(template) {
            case 'welcome':
                message = `Halo {{ $guest->name }}, selamat datang di hotel kami! Terima kasih telah mempercayakan kami untuk kebutuhan akomodasi Anda.`;
                break;
            case 'birthday':
                message = `Selamat ulang tahun {{ $guest->name }}! Semoga hari spesial Anda dipenuhi kebahagiaan. Ada penawaran khusus menanti Anda!`;
                break;
            case 'promo':
                message = `Hai {{ $guest->name }}, dapatkan diskon spesial 20% untuk booking berikutnya! Berlaku hingga akhir bulan ini.`;
                break;
            case 'feedback':
                message = `Halo {{ $guest->name }}, bagaimana pengalaman menginap Anda? Kami sangat menghargai feedback dari Anda.`;
                break;
            case 'reminder':
                message = `Hai {{ $guest->name }}, jangan lupa booking Anda besok. Kami tunggu kedatangan Anda!`;
                break;
        }
        
        $('#messageContent').val(message);
        $('#messageContent').trigger('input');
    });
}

// Updated calculatePrice function - lebih robust
function calculatePrice() {
    const roomSelected = $('#roomSelect').val();
    const roomOption = $('#roomSelect option:selected');
    
    if (!roomSelected || !roomOption.length || roomSelected === "") {
        $('#pricePreview').hide();
        $('#submitBookingBtn').prop('disabled', true);
        return;
    }
    
    // Get data from selected room option
    const roomPrice = parseFloat(roomOption.data('price')) || 0;
    const roomPpn = parseFloat(roomOption.data('ppn')) || 11;
    const roomNumber = roomOption.data('room-number') || '';
    const typeName = roomOption.data('type-name') || '';
    const capacityAdult = roomOption.data('capacity-adult') || 0;
    const capacityChild = roomOption.data('capacity-child') || 0;
    const description = roomOption.data('description') || '';
    
    const checkinDate = new Date($('#checkinDate').val());
    const checkoutDate = new Date($('#checkoutDate').val());
    
    if (!roomPrice || !checkinDate || !checkoutDate || isNaN(checkinDate) || isNaN(checkoutDate)) {
        $('#pricePreview').hide();
        $('#submitBookingBtn').prop('disabled', true);
        return;
    }
    
    // Calculate nights and pricing
    const nights = Math.ceil((checkoutDate - checkinDate) / (1000 * 60 * 60 * 24));
    
    if (nights <= 0) {
        $('#pricePreview').hide();
        $('#submitBookingBtn').prop('disabled', true);
        return;
    }
    
    const subtotal = roomPrice * nights;
    const ppnAmount = Math.round(subtotal * (roomPpn / 100));
    const grandTotal = subtotal + ppnAmount;
    
    // Create enhanced price breakdown HTML
    const priceHtml = `
        <div class="room-selection-info mb-3">
            <div class="d-flex align-items-center mb-2">
                <div class="room-type-badge bg-primary text-white px-3 py-1 rounded-pill mr-2">
                    <i class="fas fa-bed mr-1"></i>${typeName}
                </div>
                <div class="room-number-badge bg-success text-white px-2 py-1 rounded">
                    ${roomNumber}
                </div>
            </div>
            ${description ? `<small class="text-muted"><i class="fas fa-info-circle mr-1"></i>${description}</small><br>` : ''}
            <small class="text-muted">
                <i class="fas fa-users mr-1"></i>
                Kapasitas: ${capacityAdult} Dewasa, ${capacityChild} Anak
            </small>
        </div>
        <hr>
        <div class="price-breakdown">
            <div class="row">
                <div class="col-8">Harga per malam:</div>
                <div class="col-4 text-right">Rp ${roomPrice.toLocaleString('id-ID')}</div>
            </div>
            <div class="row">
                <div class="col-8">Jumlah malam:</div>
                <div class="col-4 text-right">${nights} malam</div>
            </div>
            <div class="row">
                <div class="col-8">Subtotal:</div>
                <div class="col-4 text-right">Rp ${subtotal.toLocaleString('id-ID')}</div>
            </div>
            <div class="row">
                <div class="col-8">PPN (${roomPpn}%):</div>
                <div class="col-4 text-right">Rp ${ppnAmount.toLocaleString('id-ID')}</div>
            </div>
        </div>
        <hr>
        <div class="row font-weight-bold text-success">
            <div class="col-8"><strong>Total Bayar:</strong></div>
            <div class="col-4 text-right"><strong>Rp ${grandTotal.toLocaleString('id-ID')}</strong></div>
        </div>
        <hr>
        <div class="booking-period text-center">
            <small class="text-muted">
                <i class="fas fa-calendar mr-1"></i>
                ${checkinDate.toLocaleDateString('id-ID', { 
                    weekday: 'short', 
                    day: '2-digit', 
                    month: 'short' 
                })} - ${checkoutDate.toLocaleDateString('id-ID', { 
                    weekday: 'short', 
                    day: '2-digit', 
                    month: 'short' 
                })}
            </small>
        </div>
    `;
    
    $('#priceDetails').html(priceHtml);
    $('#pricePreview').show().addClass('fade-in');
    $('#submitBookingBtn').prop('disabled', false);
    
    // Store calculated data for booking submission
    window.bookingCalculation = {
        nights: nights,
        subtotal: subtotal,
        ppnAmount: ppnAmount,
        grandTotal: grandTotal,
        roomPrice: roomPrice,
        roomPpn: roomPpn,
        roomNumber: roomNumber,
        typeName: typeName
    };
    
    // Show success toast
    showToast(`Dipilih: ${typeName} - Kamar ${roomNumber} | Total: Rp ${grandTotal.toLocaleString('id-ID')}`, 'success');
}

// Enhanced resetQuickBookingForm
function resetQuickBookingForm() {
    // Reset form
    $('#quickBookingForm')[0].reset();
    
    // Reset select options
    $('#roomSelect').html('<option value="">Pilih tanggal untuk melihat kamar tersedia...</option>').prop('disabled', true);
    
    // Hide price preview
    $('#pricePreview').hide().removeClass('fade-in');
    
    // Disable submit button
    $('#submitBookingBtn').prop('disabled', true);
    
    // Reset dates to today and tomorrow
    const today = new Date().toISOString().split('T')[0];
    $('#checkinDate').val(today);
    
    const tomorrow = new Date();
    tomorrow.setDate(tomorrow.getDate() + 1);
    $('#checkoutDate').val(tomorrow.toISOString().split('T')[0]);
    
    // Clear stored calculation
    window.bookingCalculation = null;
    
    // Set proper min dates
    $('#checkinDate').attr('min', today);
    $('#checkoutDate').attr('min', tomorrow.toISOString().split('T')[0]);
}

// Enhanced document ready function
$(document).ready(function() {
    // Set minimum dates
    const today = new Date().toISOString().split('T')[0];
    const tomorrow = new Date();
    tomorrow.setDate(tomorrow.getDate() + 1);
    
    $('#checkinDate').attr('min', today).val(today);
    $('#checkoutDate').attr('min', tomorrow.toISOString().split('T')[0]).val(tomorrow.toISOString().split('T')[0]);
    
    // Load available rooms for today-tomorrow by default
 
    // Setup event listeners
    setupEventListeners();
    
    // Initialize message templates
    initializeMessageTemplates();
});


// Utility function for toast notifications
function showToast(message, type = 'info') {
    const bgColor = {
        'success': '#28a745',
        'error': '#dc3545', 
        'warning': '#ffc107',
        'info': '#17a2b8'
    };
    
    const textColor = type === 'warning' ? '#000' : '#fff';
    
    const toast = $(`
        <div class="toast-notification" style="
            position: fixed; 
            top: 20px; 
            right: 20px; 
            z-index: 9999;
            background: ${bgColor[type] || bgColor.info};
            color: ${textColor};
            padding: 12px 20px;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            transform: translateX(100%);
            transition: transform 0.3s ease;
            max-width: 300px;
        ">
            <i class="fas fa-${type === 'success' ? 'check' : type === 'error' ? 'times' : type === 'warning' ? 'exclamation-triangle' : 'info'} mr-2"></i>
            ${message}
        </div>
    `);
    
    $('body').append(toast);
    
    // Animate in
    setTimeout(() => toast.css('transform', 'translateX(0)'), 100);
    
    // Auto remove
    setTimeout(() => {
        toast.css('transform', 'translateX(100%)');
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}
function sendMessage() {
    $('#messageModal').modal('show');
}

function submitMessage() {
    const message = $('#messageContent').val();
    const sendSMS = $('#sendSMS').is(':checked');
    const sendEmail = $('#sendEmail').is(':checked');
    
    if (!message.trim()) {
        Swal.fire('Error', 'Pesan tidak boleh kosong', 'error');
        return;
    }
    
    if (!sendSMS && !sendEmail) {
        Swal.fire('Error', 'Pilih minimal satu metode pengiriman', 'error');
        return;
    }
    
    // Show loading
    Swal.fire({
        title: 'Mengirim Pesan...',
        html: 'Mohon tunggu sebentar',
        allowOutsideClick: false,
        didOpen: () => {
            Swal.showLoading();
        }
    });
    
    // Simulate API call
    setTimeout(() => {
        let methods = [];
        if (sendSMS) methods.push('SMS');
        if (sendEmail) methods.push('Email');
        
        Swal.fire({
            icon: 'success',
            title: 'Pesan Terkirim!',
            text: `Pesan berhasil dikirim via ${methods.join(' dan ')}.`,
            timer: 3000
        });
        
        $('#messageModal').modal('hide');
        $('#messageForm')[0].reset();
    }, 1500);
}

function deleteGuest() {
    Swal.fire({
        title: 'Konfirmasi Hapus',
        html: `
            <div class="text-left">
                <p>Apakah Anda yakin ingin menghapus tamu <strong>{{ $guest->name }}</strong>?</p>
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle mr-1"></i>
                    <strong>Peringatan:</strong> Tindakan ini tidak dapat dibatalkan!
                </div>
                @if($guest->bookings->count() > 0)
                <div class="alert alert-danger">
                    <i class="fas fa-info-circle mr-1"></i>
                    Tamu ini memiliki {{ $guest->bookings->count() }} booking. 
                    Data booking akan tetap tersimpan namun tidak terhubung dengan data tamu.
                </div>
                @endif
            </div>
        `,
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: '<i class="fas fa-trash mr-1"></i> Ya, Hapus!',
        cancelButtonText: '<i class="fas fa-times mr-1"></i> Batal',
        reverseButtons: true
    }).then((result) => {
        if (result.isConfirmed) {
            $('#deleteForm').submit();
        }
    });
}

function refreshBookings() {
    Swal.fire({
        title: 'Memuat ulang...',
        timer: 1000,
        didOpen: () => {
            Swal.showLoading();
        }
    }).then(() => {
        location.reload();
    });
}

function checkinBooking(bookingId) {
    Swal.fire({
        title: 'Konfirmasi Check-in',
        text: 'Proses check-in untuk booking ini?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#28a745',
        cancelButtonColor: '#6c757d',
        confirmButtonText: 'Ya, Check-in',
        cancelButtonText: 'Batal'
    }).then((result) => {
        if (result.isConfirmed) {
            // Process check-in
            Swal.fire('Berhasil!', 'Check-in berhasil diproses', 'success');
        }
    });
}

function checkoutBooking(bookingId) {
    Swal.fire({
        title: 'Konfirmasi Check-out',
        text: 'Proses check-out untuk booking ini?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#28a745',
        cancelButtonColor: '#6c757d',
        confirmButtonText: 'Ya, Check-out',
        cancelButtonText: 'Batal'
    }).then((result) => {
        if (result.isConfirmed) {
            // Process check-out
            Swal.fire('Berhasil!', 'Check-out berhasil diproses', 'success');
        }
    });
}

function initializeMessageTemplates() {
    // Initialize any additional message template logic
}

// Handle success/error messages from server
@if(session('success'))
    Swal.fire({
        title: 'Berhasil!',
        text: '{{ session("success") }}',
        icon: 'success',
        timer: 3000,
        showConfirmButton: false
    });
@endif

@if(session('error'))
    Swal.fire({
        title: 'Error!',
        text: '{{ session("error") }}',
        icon: 'error'
    });
@endif
</script>
@endpush