<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Tabel utama untuk SP-SIPT
        Schema::create('sp_sipt', function (Blueprint $table) {
            $table->id();
            $table->string('nomor_surat')->unique()->nullable()->comment('Nomor surat pengaduan');
            $table->enum('status', ['draft', 'submitted', 'in_review', 'approved', 'rejected'])
                  ->default('draft')
                  ->comment('Status pengajuan');
            
            // Data Pemohon (Pihak Pertama)
            $table->string('pemohon_nama')->comment('Nama lengkap pemohon');
            $table->string('pemohon_nik', 16)->comment('NIK pemohon');
            $table->string('pemohon_tempat_lahir')->nullable();
            $table->date('pemohon_tanggal_lahir')->nullable();
            $table->text('pemohon_alamat')->comment('Alamat lengkap pemohon');
            $table->string('pemohon_rt', 5)->nullable();
            $table->string('pemohon_rw', 5)->nullable();
            $table->string('pemohon_kelurahan')->nullable();
            $table->string('pemohon_kecamatan')->nullable();
            $table->string('pemohon_kota')->nullable();
            $table->string('pemohon_provinsi')->nullable();
            $table->string('pemohon_kode_pos', 10)->nullable();
            $table->string('pemohon_telepon', 20)->nullable();
            $table->string('pemohon_hp', 20)->nullable();
            $table->string('pemohon_email')->nullable();
            
            // Data Pihak Kedua (Yang diadukan)
            $table->string('terlapor_nama')->comment('Nama pihak yang diadukan');
            $table->string('terlapor_nik', 16)->nullable();
            $table->string('terlapor_tempat_lahir')->nullable();
            $table->date('terlapor_tanggal_lahir')->nullable();
            $table->text('terlapor_alamat')->nullable();
            $table->string('terlapor_rt', 5)->nullable();
            $table->string('terlapor_rw', 5)->nullable();
            $table->string('terlapor_kelurahan')->nullable();
            $table->string('terlapor_kecamatan')->nullable();
            $table->string('terlapor_kota')->nullable();
            $table->string('terlapor_provinsi')->nullable();
            $table->string('terlapor_kode_pos', 10)->nullable();
            $table->string('terlapor_telepon', 20)->nullable();
            $table->string('terlapor_hp', 20)->nullable();
            
            // Data Tanah yang Dipersengketakan
            $table->enum('status_tanah', ['hak_milik', 'hak_guna_bangunan', 'hak_guna_usaha', 'hak_pakai', 'tanah_negara', 'lainnya'])
                  ->nullable()
                  ->comment('Status kepemilikan tanah');
            $table->string('status_tanah_lainnya')->nullable()->comment('Jika status tanah adalah lainnya');
            $table->string('nomor_sertifikat')->nullable();
            $table->date('tanggal_sertifikat')->nullable();
            $table->string('nama_pemilik_sertifikat')->nullable()->comment('Nama di sertifikat');
            
            // Lokasi Tanah
            $table->string('lokasi_jalan')->nullable()->comment('Nama jalan lokasi tanah');
            $table->string('lokasi_rt', 5)->nullable();
            $table->string('lokasi_rw', 5)->nullable();
            $table->string('lokasi_kelurahan')->nullable();
            $table->string('lokasi_kecamatan')->nullable();
            $table->string('lokasi_kota')->nullable();
            $table->string('lokasi_provinsi')->nullable();
            
            // Ukuran Tanah
            $table->decimal('luas_tanah', 10, 2)->nullable()->comment('Luas tanah dalam m²');
            $table->string('batas_utara')->nullable();
            $table->string('batas_selatan')->nullable();
            $table->string('batas_timur')->nullable();
            $table->string('batas_barat')->nullable();
            
            // Keterangan Pengaduan
            $table->text('kronologi')->nullable()->comment('Kronologi permasalahan');
            $table->text('tuntutan')->nullable()->comment('Tuntutan/harapan pemohon');
            $table->text('dasar_hukum')->nullable()->comment('Dasar hukum yang digunakan');
            
            // Bukti dan Dokumen (referensi ke tabel terpisah)
            // Akan dibuat di tabel sp_sipt_dokumen
            
            // Tanda Tangan dan Persetujuan
            $table->boolean('pernyataan_kebenaran')->default(false)
                  ->comment('Pernyataan bahwa data yang diberikan benar');
            $table->boolean('kesediaan_mediasi')->default(false)
                  ->comment('Kesediaan untuk mediasi');
            $table->string('ttd_pemohon_path')->nullable()->comment('Path file tanda tangan pemohon');
            $table->string('ttd_terlapor_path')->nullable()->comment('Path file tanda tangan terlapor');
            
            // Koordinat Tanah (untuk peta)
            $table->decimal('latitude', 10, 8)->nullable();
            $table->decimal('longitude', 11, 8)->nullable();
            $table->json('polygon_coordinates')->nullable()->comment('Koordinat polygon tanah dalam format JSON');
            
            // Metadata
            $table->unsignedBigInteger('created_by')->nullable();
            $table->unsignedBigInteger('updated_by')->nullable();
            $table->unsignedBigInteger('approved_by')->nullable();
            $table->timestamp('submitted_at')->nullable();
            $table->timestamp('approved_at')->nullable();
            $table->text('rejection_reason')->nullable();
            $table->timestamps();
            $table->softDeletes();
            
            // Indexes
            $table->index('nomor_surat');
            $table->index('pemohon_nik');
            $table->index('terlapor_nik');
            $table->index('nomor_sertifikat');
            $table->index('status');
            $table->index('created_at');
        });

        // Tabel untuk menyimpan dokumen pendukung
        Schema::create('sp_sipt_dokumen', function (Blueprint $table) {
            $table->id();
            $table->foreignId('sp_sipt_id')->constrained('sp_sipt')->onDelete('cascade');
            $table->enum('jenis_dokumen', [
                'ktp_pemohon',
                'ktp_terlapor',
                'sertifikat_tanah',
                'surat_ukur',
                'imb',
                'pbb',
                'akta_jual_beli',
                'surat_waris',
                'surat_hibah',
                'putusan_pengadilan',
                'berita_acara',
                'foto_lokasi',
                'dokumen_pendukung_lainnya'
            ])->comment('Jenis dokumen yang diupload');
            $table->string('nama_file');
            $table->string('path_file');
            $table->string('mime_type')->nullable();
            $table->unsignedBigInteger('ukuran_file')->nullable()->comment('Ukuran file dalam bytes');
            $table->text('keterangan')->nullable();
            $table->timestamps();
            
            $table->index(['sp_sipt_id', 'jenis_dokumen']);
        });

        // Tabel untuk menyimpan riwayat status
        Schema::create('sp_sipt_status_history', function (Blueprint $table) {
            $table->id();
            $table->foreignId('sp_sipt_id')->constrained('sp_sipt')->onDelete('cascade');
            $table->enum('status_from', ['draft', 'submitted', 'in_review', 'approved', 'rejected'])->nullable();
            $table->enum('status_to', ['draft', 'submitted', 'in_review', 'approved', 'rejected']);
            $table->text('catatan')->nullable();
            $table->unsignedBigInteger('changed_by')->nullable();
            $table->timestamp('changed_at');
            $table->timestamps();
            
            $table->index(['sp_sipt_id', 'created_at']);
        });

        // Tabel untuk menyimpan catatan/komentar dari petugas
        Schema::create('sp_sipt_catatan', function (Blueprint $table) {
            $table->id();
            $table->foreignId('sp_sipt_id')->constrained('sp_sipt')->onDelete('cascade');
            $table->text('catatan');
            $table->enum('jenis_catatan', ['internal', 'feedback', 'revisi'])->default('internal');
            $table->boolean('is_visible_to_pemohon')->default(false);
            $table->unsignedBigInteger('created_by');
            $table->timestamps();
            
            $table->index(['sp_sipt_id', 'created_at']);
        });

        // Tabel untuk menyimpan jadwal mediasi/pertemuan
        Schema::create('sp_sipt_jadwal', function (Blueprint $table) {
            $table->id();
            $table->foreignId('sp_sipt_id')->constrained('sp_sipt')->onDelete('cascade');
            $table->datetime('tanggal_waktu');
            $table->string('tempat');
            $table->enum('jenis_kegiatan', ['mediasi', 'survey', 'pertemuan', 'sidang']);
            $table->text('agenda')->nullable();
            $table->text('hasil')->nullable();
            $table->enum('status', ['scheduled', 'ongoing', 'completed', 'cancelled'])->default('scheduled');
            $table->unsignedBigInteger('created_by');
            $table->timestamps();
            
            $table->index(['sp_sipt_id', 'tanggal_waktu']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('sp_sipt_jadwal');
        Schema::dropIfExists('sp_sipt_catatan');
        Schema::dropIfExists('sp_sipt_status_history');
        Schema::dropIfExists('sp_sipt_dokumen');
        Schema::dropIfExists('sp_sipt');
    }
};