<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        Schema::table('skpt_applications', function (Blueprint $table) {
             
                // Verification related columns
                $table->enum('verification_status', ['pending', 'approved', 'needs_revision', 'rejected'])
                    ->nullable()
                    ->after('status_permohonan')
                    ->comment('Status verifikasi dokumen');

                $table->text('verification_notes')
                    ->nullable()
                    ->after('verification_status')
                    ->comment('Catatan verifikasi');

                $table->date('verification_date')
                    ->nullable()
                    ->after('verification_notes')
                    ->comment('Tanggal verifikasi');

                $table->unsignedBigInteger('verified_by')
                    ->nullable()
                    ->after('verification_date')
                    ->comment('ID user yang melakukan verifikasi');

                $table->timestamp('verified_at')
                    ->nullable()
                    ->after('verified_by')
                    ->comment('Timestamp verifikasi');

                // Document finalization columns
                $table->boolean('document_verified')
                    ->default(false)
                    ->after('verified_at')
                    ->comment('Apakah dokumen sudah diverifikasi');

                $table->boolean('document_finalized')
                    ->default(false)
                    ->after('document_verified')
                    ->comment('Apakah dokumen sudah difinalisasi');

                $table->unsignedBigInteger('finalized_by')
                    ->nullable()
                    ->after('document_finalized')
                    ->comment('ID user yang memfinalisasi dokumen');

                $table->timestamp('finalized_at')
                    ->nullable()
                    ->after('finalized_by')
                    ->comment('Timestamp finalisasi dokumen');

                // Print ready status
                $table->boolean('ready_for_print')
                    ->default(false)
                    ->after('finalized_at')
                    ->comment('Apakah dokumen siap untuk dicetak');

                $table->boolean('ready_for_official_print')
                    ->default(false)
                    ->after('ready_for_print')
                    ->comment('Apakah dokumen siap untuk cetak resmi');

                // Document status workflow
                $table->enum('document_status', ['draft', 'review', 'approved', 'rejected', 'finalized'])
                    ->default('draft')
                    ->after('ready_for_official_print')
                    ->comment('Status workflow dokumen');

                $table->text('status_notes')
                    ->nullable()
                    ->after('document_status')
                    ->comment('Catatan status dokumen');

                $table->unsignedBigInteger('status_updated_by')
                    ->nullable()
                    ->after('status_notes')
                    ->comment('ID user yang mengupdate status');

                $table->timestamp('status_updated_at')
                    ->nullable()
                    ->after('status_updated_by')
                    ->comment('Timestamp update status');

                // Add foreign key constraints if users table exists
                if (Schema::hasTable('users')) {
                    $table->foreign('verified_by')->references('id')->on('users')->onDelete('set null');
                    $table->foreign('finalized_by')->references('id')->on('users')->onDelete('set null');
                    $table->foreign('status_updated_by')->references('id')->on('users')->onDelete('set null');
                }

                // Add indexes for better performance
                $table->index(['verification_status', 'document_status']);
                $table->index(['document_verified', 'document_finalized']);
                $table->index(['ready_for_print', 'ready_for_official_print']);
                $table->index('verified_at');
                $table->index('finalized_at');
           
        });
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
       Schema::table('skpt_applications', function (Blueprint $table) {
            // Drop foreign key constraints first
            if (Schema::hasTable('users')) {
                $table->dropForeign(['verified_by']);
                $table->dropForeign(['finalized_by']);
                $table->dropForeign(['status_updated_by']);
            }

            // Drop indexes
            $table->dropIndex(['verification_status', 'document_status']);
            $table->dropIndex(['document_verified', 'document_finalized']);
            $table->dropIndex(['ready_for_print', 'ready_for_official_print']);
            $table->dropIndex(['verified_at']);
            $table->dropIndex(['finalized_at']);

            // Drop columns
            $table->dropColumn([
                'verification_status',
                'verification_notes',
                'verification_date',
                'verified_by',
                'verified_at',
                'document_verified',
                'document_finalized',
                'finalized_by',
                'finalized_at',
                'ready_for_print',
                'ready_for_official_print',
                'document_status',
                'status_notes',
                'status_updated_by',
                'status_updated_at'
            ]);
        });
    }
};
