<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Event;
use Illuminate\Auth\Events\Login;
use Illuminate\Auth\Events\Logout;
use Illuminate\Auth\Events\Failed;
use App\Models\ActivityLog;

class ActivityLogServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        $this->mergeConfigFrom(
            __DIR__.'/../../config/activity-log.php', 'activity-log'
        );
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        // Publish config file
        $this->publishes([
            __DIR__.'/../../config/activity-log.php' => config_path('activity-log.php'),
        ], 'activity-log-config');

        // Publish migration
        // $this->publishes([
        //     __DIR__.'/../../database/migrations/create_activity_logs_table.php' => database_path('migrations/'.date('Y_m_d_His', time()).'_create_activity_logs_table.php'),
        // ], 'activity-log-migrations');

        // Auto-register event listeners
        $this->registerEventListeners();
    }

    /**
     * Register event listeners for automatic logging
     */
    protected function registerEventListeners(): void
    {
        // Listen to authentication events
        Event::listen(Login::class, function ($event) {
            ActivityLog::logActivity(
                'Login',
                'User berhasil login ke sistem',
                'Authentication',
                'Berhasil'
            );
        });

        Event::listen(Logout::class, function ($event) {
            ActivityLog::logActivity(
                'Logout',
                'User berhasil logout dari sistem', 
                'Authentication',
                'Berhasil'
            );
        });

        Event::listen(Failed::class, function ($event) {
            ActivityLog::create([
                'waktu' => now(),
                'user_name' => 'Guest User',
                'user_role' => 'Guest',
                'aktivitas' => 'Failed Login',
                'deskripsi' => 'Percobaan login gagal - password/username salah',
                'module' => 'Authentication',
                'status' => 'Gagal',
                'ip_address' => request()->ip(),
                'user_id' => null,
                'browser' => request()->header('User-Agent'),
                'device' => $this->getDeviceInfo(),
                'additional_data' => ['attempted_email' => $event->credentials['email'] ?? null]
            ]);
        });
    }

    /**
     * Get device info from user agent
     */
    protected function getDeviceInfo(): string
    {
        $userAgent = request()->header('User-Agent');
        
        if (strpos($userAgent, 'Mobile') !== false) {
            return 'Mobile';
        } elseif (strpos($userAgent, 'Tablet') !== false) {
            return 'Tablet';
        } else {
            return 'Desktop';
        }
    }
}