<?php
// ==========================================
// 2. Model SpSiptDokumen.php
// ==========================================

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;

class SpSiptDokumen extends Model
{
    use HasFactory;

    protected $table = 'sp_sipt_dokumen';

    protected $fillable = [
        'sp_sipt_id',
        'jenis_dokumen',
        'nama_file',
        'path_file',
        'mime_type',
        'ukuran_file',
        'keterangan'
    ];

    /**
     * Jenis dokumen constants
     */
    const DOK_KTP_PEMOHON = 'ktp_pemohon';
    const DOK_KTP_TERLAPOR = 'ktp_terlapor';
    const DOK_SERTIFIKAT = 'sertifikat_tanah';
    const DOK_SURAT_UKUR = 'surat_ukur';
    const DOK_IMB = 'imb';
    const DOK_KWITANSI = 'kwitansi';
    const DOK_PBB = 'pbb';
    const DOK_AJB = 'akta_jual_beli';
    const DOK_SURAT_WARIS = 'surat_waris';
    const DOK_SURAT_HIBAH = 'surat_hibah';
    const DOK_PUTUSAN = 'putusan_pengadilan';
    const DOK_BERITA_ACARA = 'berita_acara';
    const DOK_FOTO = 'foto_lokasi';
    const DOK_LAINNYA = 'dokumen_pendukung_lainnya';

    /**
     * Relationships
     */
    public function spSipt()
    {
        return $this->belongsTo(SpSipt::class, 'sp_sipt_id');
    }

    /**
     * Accessors
     */
    public function getJenisDokumenLabelAttribute()
    {
        return $this->getJenisLabel();
    }

    public function getUkuranFileFormattedAttribute()
    {
        return $this->formatFileSize($this->ukuran_file);
    }

    public function getFileUrlAttribute()
    {
        return Storage::url($this->path_file);
    }

    public function getFileExistsAttribute()
    {
        return Storage::exists($this->path_file);
    }

    /**
     * Methods
     */
    public function getJenisLabel()
    {
        return match($this->jenis_dokumen) {
            self::DOK_KTP_PEMOHON => 'KTP Pemohon',
            self::DOK_KTP_TERLAPOR => 'KTP Terlapor',
            self::DOK_SERTIFIKAT => 'Sertifikat Tanah',
            self::DOK_SURAT_UKUR => 'Surat Ukur',
            self::DOK_IMB => 'IMB',
            self::DOK_PBB => 'PBB',
            self::DOK_AJB => 'Akta Jual Beli',
            self::DOK_SURAT_WARIS => 'Surat Waris',
            self::DOK_SURAT_HIBAH => 'Surat Hibah',
            self::DOK_PUTUSAN => 'Putusan Pengadilan',
            self::DOK_BERITA_ACARA => 'Berita Acara',
            self::DOK_FOTO => 'Foto Lokasi',
            self::DOK_LAINNYA => 'Dokumen Lainnya',
            default => 'Unknown'
        };
    }

    public function formatFileSize($bytes)
    {
        if ($bytes >= 1073741824) {
            return number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        } elseif ($bytes > 1) {
            return $bytes . ' bytes';
        } elseif ($bytes == 1) {
            return $bytes . ' byte';
        } else {
            return '0 bytes';
        }
    }

    public function deleteFile()
    {
        if (Storage::exists($this->path_file)) {
            Storage::delete($this->path_file);
        }
    }

    public function isImage()
    {
        return in_array($this->mime_type, [
            'image/jpeg', 
            'image/png', 
            'image/gif', 
            'image/webp',
            'image/jpg'
        ]);
    }

    public function isPdf()
    {
        return $this->mime_type === 'application/pdf';
    }

    public function getIcon()
    {
        if ($this->isImage()) {
            return 'fa-image';
        } elseif ($this->isPdf()) {
            return 'fa-file-pdf';
        } else {
            return 'fa-file';
        }
    }

    /**
     * Boot method
     */
    protected static function boot()
    {
        parent::boot();

        static::deleting(function ($dokumen) {
            $dokumen->deleteFile();
        });
    }
}

