<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;

class SKPTDocument extends Model
{
    use HasFactory;

    protected $table = 'skpt_documents';

    protected $guarded = [''];

    protected $casts = [
        'is_required' => 'boolean',
        'is_uploaded' => 'boolean'
    ];

    // Konstanta untuk jenis dokumen
    const DOCUMENT_TYPES = [
        'ktp' => 'Photo Copy KTP',
        'kk' => 'Photo Copy KK (Kartu Keluarga)',
        'ktp_saksi' => 'Photo Copy KTP Saksi Batas',
        'surat_pengantar_rt' => 'Surat Pengantar dari RT',
        'surat_pengantar_kampung' => 'Surat Pengantar dari Kepala Kampung',
        'copy_surat_riwayat' => 'Copy Surat Riwayat Peguasaan Fisik Bidang Tanah',
        'surat_keterangan' => 'Surat Keterangan Kematian/Pembagian Waris (Optional)',
        'sppt_pbb' => 'SPPT PBB (Optional)',
        'surat_permohonan' => 'Surat Permohonan SKPT (sudah ditandatangani)',
        'foto_bukti_1' => 'Foto copy bukti yuridis penguasaan tanah negara hal-1',
        'foto_bukti_2' => 'Foto copy bukti yuridis penguasaan tanah negara hal-2',
    ];


    // Relasi dengan user yang memverifikasi
    public function verifiedBy()
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    // Accessor untuk mendapatkan nama status verifikasi
    public function getVerificationStatusNameAttribute()
    {
        return self::VERIFICATION_STATUS[$this->verification_status] ?? $this->verification_status;
    }

    // Accessor untuk mendapatkan badge class berdasarkan status
    public function getVerificationBadgeClassAttribute()
    {
        $classes = [
            'PENDING' => 'bg-yellow-100 text-yellow-800',
            'SUDAH' => 'bg-green-100 text-green-800',
            'DITOLAK' => 'bg-red-100 text-red-800'
        ];
        
        return $classes[$this->verification_status] ?? 'bg-gray-100 text-gray-800';
    }

    // Scope untuk filter berdasarkan status verifikasi
    public function scopeByVerificationStatus($query, $status)
    {
        return $query->where('verification_status', $status);
    }

    // Scope untuk dokumen yang sudah diverifikasi
    public function scopeVerified($query)
    {
        return $query->where('verification_status', 'SUDAH');
    }

    // Scope untuk dokumen yang masih pending
    public function scopePending($query)
    {
        return $query->where('verification_status', 'PENDING');
    }

    // Scope untuk dokumen yang ditolak
    public function scopeRejected($query)
    {
        return $query->where('verification_status', 'DITOLAK');
    }

    // Scope untuk dokumen wajib
    public function scopeRequired($query)
    {
        return $query->where('is_required', true);
    }

    // Scope untuk dokumen opsional
    public function scopeOptional($query)
    {
        return $query->where('is_required', false);
    }

    // Scope untuk dokumen tambahan
    public function scopeAdditional($query)
    {
        return $query->where('is_additional', true);
    }

    // Method untuk verifikasi dokumen
    public function verify($userId, $notes = null)
    {
        $this->update([
            'verification_status' => 'SUDAH',
            'verification_notes' => $notes,
            'verified_at' => now(),
            'verified_by' => $userId
        ]);
    }

    // Method untuk menolak dokumen
    public function reject($userId, $notes = null)
    {
        $this->update([
            'verification_status' => 'DITOLAK',
            'verification_notes' => $notes,
            'verified_at' => now(),
            'verified_by' => $userId
        ]);
    }

    // Method untuk reset verifikasi
    public function resetVerification()
    {
        $this->update([
            'verification_status' => 'PENDING',
            'verification_notes' => null,
            'verified_at' => null,
            'verified_by' => null
        ]);
    }

    // Accessor untuk mendapatkan nama dokumen yang readable
    public function getDocumentTypeNameAttribute()
    {
        return self::DOCUMENT_TYPES[$this->document_type] ?? $this->document_type;
    }

    // Accessor untuk mendapatkan URL file
    public function getFileUrlAttribute()
    {
        if ($this->file_path && Storage::exists($this->file_path)) {
            return Storage::url($this->file_path);
        }
        return null;
    }

    // Accessor untuk format ukuran file
    public function getFileSizeFormattedAttribute()
    {
        $bytes = (int) $this->file_size;
        
        if ($bytes >= 1073741824) {
            return number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        } elseif ($bytes > 1) {
            return $bytes . ' bytes';
        } elseif ($bytes == 1) {
            return $bytes . ' byte';
        } else {
            return '0 bytes';
        }
    }

    // Method untuk check apakah file ada
    public function fileExists()
    {
        return $this->file_path && Storage::exists($this->file_path);
    }

    // Method untuk delete file dari storage
    public function deleteFile()
    {
        if ($this->fileExists()) {
            Storage::delete($this->file_path);
        }
    }

    // Method untuk mendapatkan icon berdasarkan ekstensi
    public function getFileIconAttribute()
    {
        $icons = [
            'pdf' => 'fas fa-file-pdf text-red-500',
            'doc' => 'fas fa-file-word text-blue-500',
            'docx' => 'fas fa-file-word text-blue-500',
            'jpg' => 'fas fa-file-image text-green-500',
            'jpeg' => 'fas fa-file-image text-green-500',
            'png' => 'fas fa-file-image text-green-500',
            'gif' => 'fas fa-file-image text-green-500',
            'xls' => 'fas fa-file-excel text-green-600',
            'xlsx' => 'fas fa-file-excel text-green-600',
        ];

        return $icons[strtolower($this->file_extension)] ?? 'fas fa-file text-gray-500';
    }

    // Method untuk validasi ekstensi file
    public static function getAllowedExtensions()
    {
        return ['pdf', 'jpg', 'jpeg', 'png', 'doc', 'docx'];
    }

    // Method untuk validasi ukuran file (dalam bytes)
    public static function getMaxFileSize()
    {
        return 5 * 1024 * 1024; // 5MB
    }

    // Boot method
    protected static function boot()
    {
        parent::boot();
        
        // Auto update status upload
        static::creating(function ($model) {
            if ($model->file_path) {
                $model->is_uploaded = true;
            }
        });

        static::updating(function ($model) {
            if ($model->file_path) {
                $model->is_uploaded = true;
            }
        });

        // Delete file saat model dihapus
        static::deleting(function ($model) {
            $model->deleteFile();
        });
    }
}