<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;

class SKPTAdditionalDocument extends Model
{
    use HasFactory;

    protected $table = 'skpt_additional_documents';

    protected $fillable = [
        'skpt_application_id',
        'document_name',
        'document_description',
        'file_path',
        'original_filename',
        'file_size',
        'file_extension',
        'verification_status',
        'verification_notes',
        'verified_at',
        'verified_by',
        'is_active'
    ];

    protected $casts = [
        'verified_at' => 'datetime',
        'is_active' => 'boolean'
    ];

    // Konstanta untuk status verifikasi
    const VERIFICATION_STATUS = [
        'PENDING' => 'Menunggu Verifikasi',
        'SUDAH' => 'Sudah Terverifikasi',
        'DITOLAK' => 'Ditolak'
    ];

    // Relasi dengan aplikasi SKPT
    public function skptApplication()
    {
        return $this->belongsTo(SKPTApplication::class, 'skpt_application_id');
    }
    
    // Relasi dengan user yang memverifikasi
    public function verifiedBy()
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    // Accessor untuk mendapatkan URL file
    public function getFileUrlAttribute()
    {
        if ($this->file_path && Storage::exists($this->file_path)) {
            return Storage::url($this->file_path);
        }
        return null;
    }

    // Accessor untuk format ukuran file
    public function getFileSizeFormattedAttribute()
    {
        $bytes = (int) $this->file_size;
        
        if ($bytes >= 1073741824) {
            return number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        } elseif ($bytes > 1) {
            return $bytes . ' bytes';
        } elseif ($bytes == 1) {
            return $bytes . ' byte';
        } else {
            return '0 bytes';
        }
    }

    // Accessor untuk mendapatkan nama status verifikasi
    public function getVerificationStatusNameAttribute()
    {
        return self::VERIFICATION_STATUS[$this->verification_status] ?? $this->verification_status;
    }

    // Accessor untuk mendapatkan badge class berdasarkan status
    public function getVerificationBadgeClassAttribute()
    {
        $classes = [
            'PENDING' => 'bg-yellow-100 text-yellow-800',
            'SUDAH' => 'bg-green-100 text-green-800',
            'DITOLAK' => 'bg-red-100 text-red-800'
        ];
        
        return $classes[$this->verification_status] ?? 'bg-gray-100 text-gray-800';
    }

    // Method untuk mendapatkan icon berdasarkan ekstensi
    public function getFileIconAttribute()
    {
        $icons = [
            'pdf' => 'fas fa-file-pdf text-red-500',
            'doc' => 'fas fa-file-word text-blue-500',
            'docx' => 'fas fa-file-word text-blue-500',
            'jpg' => 'fas fa-file-image text-green-500',
            'jpeg' => 'fas fa-file-image text-green-500',
            'png' => 'fas fa-file-image text-green-500',
            'gif' => 'fas fa-file-image text-green-500',
            'xls' => 'fas fa-file-excel text-green-600',
            'xlsx' => 'fas fa-file-excel text-green-600',
        ];

        return $icons[strtolower($this->file_extension)] ?? 'fas fa-file text-gray-500';
    }

    // Method untuk check apakah file ada
    public function fileExists()
    {
        return $this->file_path && Storage::exists($this->file_path);
    }

    // Method untuk delete file dari storage
    public function deleteFile()
    {
        if ($this->fileExists()) {
            Storage::delete($this->file_path);
        }
    }

    // Scope untuk filter berdasarkan status verifikasi
    public function scopeByVerificationStatus($query, $status)
    {
        return $query->where('verification_status', $status);
    }

    // Scope untuk dokumen aktif
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    // Scope untuk dokumen yang sudah diverifikasi
    public function scopeVerified($query)
    {
        return $query->where('verification_status', 'SUDAH');
    }

    // Scope untuk dokumen yang masih pending
    public function scopePending($query)
    {
        return $query->where('verification_status', 'PENDING');
    }

    // Scope untuk dokumen yang ditolak
    public function scopeRejected($query)
    {
        return $query->where('verification_status', 'DITOLAK');
    }

    // Method untuk verifikasi dokumen
    public function verify($userId, $notes = null)
    {
        $this->update([
            'verification_status' => 'SUDAH',
            'verification_notes' => $notes,
            'verified_at' => now(),
            'verified_by' => $userId
        ]);
    }

    // Method untuk menolak dokumen
    public function reject($userId, $notes = null)
    {
        $this->update([
            'verification_status' => 'DITOLAK',
            'verification_notes' => $notes,
            'verified_at' => now(),
            'verified_by' => $userId
        ]);
    }

    // Method untuk reset verifikasi
    public function resetVerification()
    {
        $this->update([
            'verification_status' => 'PENDING',
            'verification_notes' => null,
            'verified_at' => null,
            'verified_by' => null
        ]);
    }

    // Method untuk soft delete
    public function softDelete()
    {
        $this->update(['is_active' => false]);
    }

    // Method untuk restore
    public function restore()
    {
        $this->update(['is_active' => true]);
    }

    // Method untuk validasi ekstensi file
    public static function getAllowedExtensions()
    {
        return ['pdf', 'jpg', 'jpeg', 'png', 'doc', 'docx'];
    }

    // Method untuk validasi ukuran file (dalam bytes)
    public static function getMaxFileSize()
    {
        return 5 * 1024 * 1024; // 5MB
    }

    // Boot method
    protected static function boot()
    {
        parent::boot();
        
        // Delete file saat model dihapus
        static::deleting(function ($model) {
            $model->deleteFile();
        });
    }
}