<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Wildside\Userstamps\Userstamps;
use Illuminate\Database\Eloquent\SoftDeletes;

class MsTypeRooms extends Model
{
    use HasFactory, Userstamps, SoftDeletes;
    
    protected $guarded = [];
    protected $table = 'ms_type_rooms';
    protected $casts = [
        'harga_per_malam' => 'decimal:2',
        'harga_per_orang' => 'decimal:2',
        'harga_per_jam' => 'decimal:2',
        'ppn' => 'decimal:2',
    ];

    // Relationships
    public function rooms()
    {
        return $this->hasMany(MsRooms::class, 'type_room_id');
    }

    // Accessors
    public function getHargaPerMalamFormattedAttribute()
    {
        return 'Rp ' . number_format($this->harga_per_malam, 0, ',', '.');
    }

    public function getHargaPerOrangFormattedAttribute()
    {
        return 'Rp ' . number_format($this->harga_per_orang, 0, ',', '.');
    }

    public function getHargaPerJamFormattedAttribute()
    {
        return 'Rp ' . number_format($this->harga_per_jam, 0, ',', '.');
    }

    // NEW: Get available booking types for this room type
    public function getAvailableBookingTypesAttribute()
    {
        $types = [];
        
        if ($this->harga_per_malam > 0) {
            $types[] = [
                'type' => 'daily',
                'name' => 'Harian',
                'price' => $this->harga_per_malam,
                'formatted_price' => $this->harga_per_malam_formatted,
                'description' => 'Per malam'
            ];
        }
        
        if ($this->harga_per_jam > 0) {
            $types[] = [
                'type' => 'hourly',
                'name' => 'Per Jam',
                'price' => $this->harga_per_jam,
                'formatted_price' => $this->harga_per_jam_formatted,
                'description' => 'Per jam'
            ];
        }
        
        if ($this->harga_per_orang > 0) {
            $types[] = [
                'type' => 'per_person',
                'name' => 'Per Orang',
                'price' => $this->harga_per_orang,
                'formatted_price' => $this->harga_per_orang_formatted,
                'description' => 'Per orang per malam'
            ];
        }
        
        return $types;
    }

    // NEW: Check if booking type is available
    public function hasBookingType($type)
    {
        switch ($type) {
            case 'daily':
                return $this->harga_per_malam > 0;
            case 'hourly':
                return $this->harga_per_jam > 0;
            case 'per_person':
                return $this->harga_per_orang > 0;
            default:
                return false;
        }
    }

    // NEW: Get price by booking type
    public function getPriceByType($type)
    {
        switch ($type) {
            case 'daily':
                return $this->harga_per_malam;
            case 'hourly':
                return $this->harga_per_jam;
            case 'per_person':
                return $this->harga_per_orang;
            default:
                return 0;
        }
    }

    // Methods - Updated untuk berbagai tipe booking
    public function calculateTotalPrice($nights = 1, $persons = 1, $hours = 1, $bookingType = 'daily')
    {
        $basePrice = $this->getPriceByType($bookingType);
        $totalPrice = 0;
        
        switch ($bookingType) {
            case 'daily':
                $totalPrice = $basePrice * $nights;
                break;
            case 'hourly':
                $totalPrice = $basePrice * $hours;
                break;
            case 'per_person':
                $totalPrice = $basePrice * $persons * $nights;
                break;
        }
        
        $ppn = $totalPrice * ($this->ppn / 100);
        
        return [
            'booking_type' => $bookingType,
            'base_price' => $basePrice,
            'total_price' => $totalPrice,
            'ppn_rate' => $this->ppn / 100,
            'ppn_amount' => $ppn,
            'grand_total' => $totalPrice + $ppn,
            'calculation_details' => [
                'nights' => $nights,
                'persons' => $persons,
                'hours' => $hours
            ]
        ];
    }

    // NEW: Get minimum booking duration for hourly bookings
    public function getMinHourlyDurationAttribute()
    {
        return 3; // Default minimum 3 jam untuk booking hourly
    }

    // NEW: Get maximum booking duration for hourly bookings
    public function getMaxHourlyDurationAttribute()
    {
        return 24; // Default maksimum 24 jam untuk booking hourly
    }

    // NEW: Validate booking parameters
    public function validateBookingParameters($bookingType, $nights = 1, $persons = 1, $hours = 1)
    {
        $errors = [];
        
        if (!$this->hasBookingType($bookingType)) {
            $errors[] = "Tipe booking {$bookingType} tidak tersedia untuk tipe kamar ini";
        }
        
        switch ($bookingType) {
            case 'daily':
                if ($nights < 1) {
                    $errors[] = "Minimal booking 1 malam untuk tipe harian";
                }
                break;
                
            case 'hourly':
                if ($hours < $this->min_hourly_duration) {
                    $errors[] = "Minimal booking {$this->min_hourly_duration} jam untuk tipe per jam";
                }
                if ($hours > $this->max_hourly_duration) {
                    $errors[] = "Maksimal booking {$this->max_hourly_duration} jam untuk tipe per jam";
                }
                break;
                
            case 'per_person':
                if ($persons < 1) {
                    $errors[] = "Minimal 1 orang untuk tipe per orang";
                }
                if ($nights < 1) {
                    $errors[] = "Minimal booking 1 malam untuk tipe per orang";
                }
                break;
        }
        
        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }

    // Legacy method for backward compatibility
    public function calculateTotalPriceLegacy($nights, $persons = 1)
    {
        return $this->calculateTotalPrice($nights, $persons, 1, 'daily');
    }
}