<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Wildside\Userstamps\Userstamps;
use Illuminate\Database\Eloquent\SoftDeletes;

class MsRooms extends Model
{
    use HasFactory, Userstamps, SoftDeletes;
    
    protected $guarded = [];
    protected $table = 'ms_rooms';

    const STATUS_AVAILABLE = 'available';
    const STATUS_OCCUPIED = 'occupied';
    const STATUS_MAINTENANCE = 'maintenance';
    const STATUS_CLEANING = 'cleaning';

    // Relationships
    public function typeRoom()
    {
        return $this->belongsTo(MsTypeRooms::class, 'type_room_id');
    }

    public function bookingDetails()
    {
        return $this->hasMany(BookingDetail::class, 'room_id');
    }

    public function bookings()
    {
        return $this->hasManyThrough(Booking::class, BookingDetail::class, 'room_id', 'id', 'id', 'booking_id');
    }

    public function housekeepingTasks()
    {
        return $this->hasMany(HousekeepingTask::class, 'room_id');
    }

    // Get active housekeeping tasks (not completed/cancelled)
    public function activeHousekeepingTasks()
    {
        return $this->hasMany(HousekeepingTask::class, 'room_id')
                    ->whereIn('status', ['pending', 'in_progress'])
                    ->whereNull('deleted_at');
    }

    // Accessors
    public function getStatusBadgeAttribute()
    {
        $badges = [
            'available' => '<span class="badge bg-success">Tersedia</span>',
            'occupied' => '<span class="badge bg-danger">Terisi</span>',
            'maintenance' => '<span class="badge bg-warning">Maintenance</span>',
            'cleaning' => '<span class="badge bg-info">Cleaning</span>',
        ];

        return $badges[$this->status] ?? '<span class="badge bg-secondary">Unknown</span>';
    }

    public function getStatusColorAttribute()
    {
        $colors = [
            'available' => 'success',
            'occupied' => 'danger',
            'maintenance' => 'warning',
            'cleaning' => 'info',
        ];

        return $colors[$this->status] ?? 'secondary';
    }

    // Scopes
    public function scopeAvailable($query)
    {
        return $query->where('status', 'available');
    }

    public function scopeOccupied($query)
    {
        return $query->where('status', 'occupied');
    }

    public function scopeMaintenance($query)
    {
        return $query->where('status', 'maintenance');
    }

    public function scopeCleaning($query)
    {
        return $query->where('status', 'cleaning');
    }

    // Status Check Methods
    public function isAvailable()
    {
        return $this->status === 'available';
    }

    public function isOccupied()
    {
        return $this->status === 'occupied';
    }

    public function isMaintenance()
    {
        return $this->status === 'maintenance';
    }

    public function isCleaning()
    {
        return $this->status === 'cleaning';
    }

    // Status Update Methods (Enhanced)
    public function setOccupied($reason = null)
    {
        // dd('tes');
        $this->update([
            'status' => 'occupied'
        ]);
        
        \Log::info("Room {$this->room_number} status changed to OCCUPIED" . ($reason ? " - Reason: $reason" : ""));
        return $this;
    }

    public function setAvailable($reason = null)
    {
        $this->update([
            'status' => 'available'
        ]);
        
        \Log::info("Room {$this->room_number} status changed to AVAILABLE" . ($reason ? " - Reason: $reason" : ""));
        return $this;
    }

    public function setMaintenance($reason = null)
    {
        $this->update([
            'status' => 'maintenance'
        ]);
        
        \Log::info("Room {$this->room_number} status changed to MAINTENANCE" . ($reason ? " - Reason: $reason" : ""));
        return $this;
    }

    public function setCleaning($reason = null)
    {
        $this->update([
            'status' => 'cleaning'
        ]);
        
        \Log::info("Room {$this->room_number} status changed to CLEANING" . ($reason ? " - Reason: $reason" : ""));
        return $this;
    }

    // NEW: Auto-update status based on housekeeping tasks
    public function updateStatusBasedOnHousekeeping()
    {
        // Get active housekeeping tasks
        $activeTasks = $this->activeHousekeepingTasks()->get();
        
        if ($activeTasks->isEmpty()) {
            // No active tasks - check if room should be available
            $this->autoSetAvailableIfPossible();
            return $this;
        }

        // Check for maintenance tasks
        $maintenanceTasks = $activeTasks->whereIn('task_type', ['maintenance']);
        if ($maintenanceTasks->isNotEmpty()) {
            if (!$this->isMaintenance()) {
                $this->setMaintenance('Housekeeping maintenance task active');
            }
            return $this;
        }

        // Check for cleaning tasks
        $cleaningTasks = $activeTasks->whereIn('task_type', ['checkout_cleaning', 'deep_cleaning']);
        if ($cleaningTasks->isNotEmpty()) {
            if (!$this->isCleaning()) {
                $this->setCleaning('Housekeeping cleaning task active');
            }
            return $this;
        }

        // If only inspection tasks, keep current status or set cleaning if available
        $inspectionTasks = $activeTasks->where('task_type', 'inspection');
        if ($inspectionTasks->isNotEmpty() && $this->isAvailable()) {
            $this->setCleaning('Room inspection in progress');
        }

        return $this;
    }

    // Auto set available if no blocking conditions
    public function autoSetAvailableIfPossible()
    {
        // Don't auto-set to available if occupied
        if ($this->isOccupied()) {
            return $this;
        }

        // Check if there are any active bookings
        $activeBookings = $this->bookings()
            ->whereIn('status', ['checked_in'])
            ->count();

        if ($activeBookings > 0) {
            $this->setOccupied('Active booking found');
            return $this;
        }

        // Check for pending check-ins today
        $todayCheckIns = $this->bookings()
            ->where('status', 'booked')
            ->whereDate('check_in_date', today())
            ->count();

        if ($todayCheckIns > 0) {
            // Keep current status or set available but log
            \Log::info("Room {$this->room_number} has check-ins today but no active housekeeping");
        }

        // No blocking conditions - set available
        if (!$this->isAvailable()) {
            $this->setAvailable('No active housekeeping tasks or bookings');
        }

        return $this;
    }

    // Check if room needs housekeeping
    public function needsHousekeeping()
    {
        // Check if there are recent checkouts without cleaning
        $recentCheckouts = $this->bookings()
            ->where('status', 'checked_out')
            ->where('updated_at', '>=', now()->subHours(24))
            ->count();

        if ($recentCheckouts > 0) {
            $cleaningTasks = $this->housekeepingTasks()
                ->where('task_type', 'checkout_cleaning')
                ->where('created_at', '>=', now()->subHours(24))
                ->where('status', 'completed')
                ->count();

            return $cleaningTasks === 0;
        }

        return false;
    }

    // Get room status for display
    public function getStatusDisplayAttribute()
    {
        $statusTexts = [
            'available' => 'Tersedia',
            'occupied' => 'Terisi',
            'maintenance' => 'Maintenance',
            'cleaning' => 'Cleaning',
        ];

        return $statusTexts[$this->status] ?? 'Unknown';
    }

    // Boot method to handle events
    protected static function boot()
    {
        parent::boot();

        // Auto-update status when room is updated
        static::updated(function ($room) {
            // If status was manually changed, don't auto-update
            if ($room->wasChanged('status')) {
                \Log::info("Room {$room->room_number} status manually changed to {$room->status}");
            }
        });
    }
}