<?php

namespace App\Http\Controllers;

use App\Models\BeritaAcaraPeninjauan;
use Illuminate\Support\Facades\Auth;

use App\Models\Kampung;
use App\Models\Peninjauan;
use App\Models\SKPTApplication;
use App\Models\SKPTDocument;
use App\Models\SKPTAdditionalDocument;
use App\Models\SkptAnak;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;


class SKPTController extends Controller
{
    /**
     * Display a listing of SKPT applications.
     */
    public function __construct()
    {
        $this->middleware('permission:skpt.view', ['only' => ['index', 'show', 'getTableData', 'getTotalCount', 'getPolygonData', 'getDetailData', 'filterSkpt']]);
        $this->middleware('permission:skpt.create', ['only' => ['create', 'store']]);
        $this->middleware('permission:skpt.edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:skpt.delete', ['only' => ['destroy', 'deleteAdditionalDocument']]);
        $this->middleware('permission:skpt.upload-document', ['only' => ['uploadDocument', 'uploadAdditionalDocument']]);
        $this->middleware('permission:skpt.verify-document', ['only' => ['verifyDocument', 'verifyAdditionalDocument']]);
        $this->middleware('permission:skpt.update-status', ['only' => ['updateStatus']]);
        $this->middleware('permission:skpt.preview', ['only' => ['preview']]);
        $this->middleware('permission:skpt.print', ['only' => ['print', 'downloadDocument', 'downloadAdditionalDocument']]);
    }
    public function cekByRegistrasi(Request $request)
    {
        $validated = $request->validate([
            'nomor_registrasi' => 'required|string|max:100',
        ]);

        $nomor = trim($validated['nomor_registrasi']);

        $skpt = SKPTApplication::where('no_registrasi', $nomor)->first();

        if (!$skpt) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak ditemukan.',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'nomor_registrasi' => $skpt->no_registrasi,
                'status' => strtoupper($skpt->status_permohonan ?? 'PENINJAUAN'),
            ],
        ]);
    }
    public function index(Request $request)
    {
        $query = SKPTApplication::with(['documents', 'additionalDocuments']);

        // Filter berdasarkan tahun
        if ($request->filled('year')) {
            $query->whereYear('tgl_penerimaan', $request->year);
        }

        // Filter berdasarkan status permohonan
        if ($request->filled('status')) {
            $query->where('status_permohonan', $request->status);
        }

        // Filter berdasarkan status kelengkapan dokumen
        if ($request->filled('document_status')) {
            $query->where('status_kelengkapan_dokumen', $request->document_status);
        }

        // Search functionality
        if ($request->filled('search')) {
            $query->search($request->search);
        }

        if (auth()->user()->hasRole('guest')) {
            $query->where('created_by', auth()->id());
        }

        // Ordering
        $query->orderBy('tgl_penerimaan', 'desc');

        $applications = $query->paginate(10)->withQueryString();

        // Data untuk filter
        $years = SKPTApplication::selectRaw('YEAR(tgl_penerimaan) as year')
                               ->distinct()
                               ->orderBy('year', 'desc')
                               ->pluck('year');

        return view('admin.skpt.index', compact('applications', 'years'));
    }

    /**
     * Show the form for creating a new SKPT application.
     */
    public function create()
    {
        $data_kampung = Kampung::all();
        return view('admin.skpt.create', compact('data_kampung'));
    }

    /**
     * Store a newly created SKPT application.
     */
    public function store(Request $request)
    {
        // Validasi data dan dokumen
        $validatedData = $request->validate([
            // Data Pemohon
            'nama_pemohon' => 'required|string|max:255',
            // 'nik' => 'required|string|max:20|unique:skpt_applications,nik',
            'no_ktp' => 'required|string|max:20|unique:skpt_applications,no_ktp',
            'tempat_lahir' => 'required|string|max:255',
            'tanggal_lahir' => 'required|date',
            'email' => 'nullable|email|max:255',
            'telepon' => 'nullable|string|max:15',
            'pekerjaan' => 'required|string|max:255',
            'tgl_penerimaan' => 'required|date',
            'tgl_permohonan' => 'required|date',

            // Alamat Pemohon
            'alamat_rt' => 'required|string|max:3',
            'alamat_rw' => 'required|string|max:3',
            'alamat_kode_pos' => 'required|string|max:5',

            // Detail Tanah
            'letak_tanah' => 'required|string',
            'tanah_rt' => 'required|string|max:3',
            'tanah_rw' => 'required|string|max:3',
            'tanah_kode_pos' => 'required|string|max:5',
            'dusun' => 'required|string|max:255',
            'kelurahan_kampung' => 'required|string|max:255',
            'peruntukan' => 'required|in:Perkebunan,Pertanian,Perumahan,Industri,Lainnya',
            'luas' => 'required|numeric|min:0',
            'penanda_tanah' => 'required|in:Batas Pagar,Batas Alam,Batas Bangunan,Lainnya',
            'utara' => 'required|in:Tanah Kosong,Rumah,Kebun,Jalan,Sungai,Lainnya',
            'timur' => 'required|in:Tanah Kosong,Rumah,Kebun,Jalan,Sungai,Lainnya',
            'selatan' => 'required|in:Jalan Kampung,Rumah,Kebun,Tanah Kosong,Sungai,Lainnya',
            'barat' => 'required|in:Tanah Hasan,Rumah,Kebun,Tanah Kosong,Sungai,Lainnya',

            // Validasi dokumen wajib
            'ktp' => 'required|file|mimes:pdf,jpg,png|max:2048',
            'kk' => 'required|file|mimes:pdf,jpg,png|max:2048',
            'ktp_saksi' => 'required|file|mimes:pdf,jpg,png|max:2048',
            'surat_pengantar_rt' => 'required|file|mimes:pdf,jpg,png|max:2048',
            'surat_pengantar_kampung' => 'required|file|mimes:pdf,jpg,png|max:2048',
            'copy_surat_riwayat' => 'required|file|mimes:pdf,jpg,png|max:2048',
            'surat_permohonan' => 'required|file|mimes:pdf,jpg,png|max:2048',
            'foto_bukti_1' => 'required|file|mimes:pdf,jpg,png|max:2048',
            'foto_bukti_2' => 'required|file|mimes:pdf,jpg,png|max:2048',

            // Validasi dokumen opsional
            'surat_keterangan' => 'nullable|file|mimes:pdf,jpg,png|max:2048',
            'sppt_pbb' => 'nullable|file|mimes:pdf,jpg,png|max:2048',
            'additional_documents.*' => 'nullable|file|mimes:pdf,jpg,png|max:2048',
            'additional_descriptions.*' => 'nullable|string|max:255',
        ]);

        try {
            DB::beginTransaction();

            // Sinkronkan no_ktp dengan nik jika nik tidak diisi secara manual
            if (empty($validatedData['nik'])) {
                $validatedData['nik'] = $validatedData['no_ktp'];
            }
            $validatedData['created_by'] = auth()->id();

            // Create SKPT Application
            $application = SKPTApplication::create($validatedData);

            // Daftar dokumen dengan status wajib/opsional
            $documents = [
                'ktp' => ['name' => 'Photo Copy KTP', 'is_required' => true, 'is_additional' => false],
                'kk' => ['name' => 'Photo Copy KK (Kartu Keluarga)', 'is_required' => true, 'is_additional' => false],
                'ktp_saksi' => ['name' => 'Photo Copy KTP Saksi Batas', 'is_required' => true, 'is_additional' => false],
                'surat_pengantar_rt' => ['name' => 'Surat Pengantar dari RT', 'is_required' => true, 'is_additional' => false],
                'surat_pengantar_kampung' => ['name' => 'Surat Pengantar dari Kepala Kampung', 'is_required' => true, 'is_additional' => false],
                'copy_surat_riwayat' => ['name' => 'Copy Surat Riwayat Peguasaan Fisik Bidang Tanah', 'is_required' => true, 'is_additional' => false],
                'surat_permohonan' => ['name' => 'Surat Permohonan SKPT', 'is_required' => true, 'is_additional' => false],
                'foto_bukti_1' => ['name' => 'Foto copy bukti yuridis penguasaan tanah negara hal-1', 'is_required' => true, 'is_additional' => false],
                'foto_bukti_2' => ['name' => 'Foto copy bukti yuridis penguasaan tanah negara hal-2', 'is_required' => true, 'is_additional' => false],
                'surat_keterangan' => ['name' => 'Surat Keterangan Kematian/Pembagian Waris', 'is_required' => false, 'is_additional' => true],
                'sppt_pbb' => ['name' => 'SPPT PBB', 'is_required' => false, 'is_additional' => true],
            ];

            //simpan data anak jika ada
            if ($request->has('anak')) {
                foreach ($request->anak as $anak) {
                    SkptAnak::create([
                        'skpt_id'   => $application->id,
                        'nama_anak' => $anak['nama_anak'],
                        'usia'      => $anak['usia'] ?? null,
                    ]);
                }
            }

            // Simpan dokumen ke storage dan database
            foreach ($documents as $key => $doc) {
                if ($request->hasFile($key)) {
                    $file = $request->file($key);
                    $fileName = time() . '_' . $key . '.' . $file->getClientOriginalExtension();
                    $filePath = $file->storeAs('documents/skpt/' . $application->id, $fileName, 'public');
                    SKPTDocument::create([
                        'skpt_application_id' => $application->id,
                        'document_type' => $key,
                        'document_name' => $doc['name'],
                        'file_path' => $filePath,
                        'file_size' => $file->getSize(),
                        'file_extension' => $file->getClientOriginalExtension(),
                        'is_required' => $doc['is_required'],
                        'is_additional' => $doc['is_additional'],
                        'is_uploaded' => true,
                        'verification_status' => 'PENDING',
                        'verification_notes' => null,
                        'verified_at' => null,
                        'verified_by' => null,
                    ]);
                } elseif ($doc['is_required']) {
                    return redirect()->back()->withInput()->with('error', 'Dokumen wajib ' . $doc['name'] . ' belum diunggah.');
                }
            }

            // Simpan dokumen tambahan jika ada
            if ($request->hasFile('additional_documents')) {
                foreach ($request->file('additional_documents') as $index => $file) {
                    if ($file) {
                        $fileName = time() . '_additional_' . $index . '.' . $file->getClientOriginalExtension();
                        $filePath = $file->storeAs('documents/skpt/' . $application->id, $fileName, 'public');
                        $description = $request->input('additional_descriptions')[$index] ?? '';
                        SKPTDocument::create([
                            'skpt_application_id' => $application->id,
                            'document_type' => 'additional_' . $index,
                            'document_name' => 'Dokumen Tambahan ' . ($index + 1),
                            'description' => $description,
                            'file_path' => $filePath,
                            'file_size' => $file->getSize(),
                            'file_extension' => $file->getClientOriginalExtension(),
                            'is_required' => false,
                            'is_additional' => true,
                            'is_uploaded' => true,
                            'verification_status' => 'PENDING',
                            'verification_notes' => null,
                            'verified_at' => null,
                            'verified_by' => null,
                        ]);
                    }
                }
            }

            // Hapus panggilan ke seeder jika tidak diperlukan
            // SKPTDocumentSeeder::createDocumentTemplatesForApplication($application->id);

            DB::commit();

            return redirect()->route('skpt.show', $application)
                        ->with('success', 'Permohonan SKPT berhasil dibuat dengan No. Registrasi: ' . $application->no_registrasi);

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                        ->withInput()
                        ->with('error', 'Terjadi kesalahan saat menyimpan data: ' . $e->getMessage());
        }
    }
    public function storeDocuments(Request $request)
    {
        $validatedData = $request->validate([
            'additional_document' => 'required|file|mimes:pdf,jpg,png|max:2048',
            'document_name' => 'required|string|max:255',
        ]);

        try {
            DB::beginTransaction();

            // Asumsikan application_id dikirim dari frontend atau diambil dari session
            // Untuk contoh ini, kita asumsikan application_id dikirim melalui form atau route
            $applicationId = $request->input('application_id'); // Anda perlu menyesuaikan ini
            if (!$applicationId) {
                throw new \Exception('ID aplikasi tidak ditemukan');
            }

            $file = $request->file('additional_document');
            $fileName = time() . '_additional_' . $file->getClientOriginalName();
            $filePath = $file->storeAs('documents/skpt/' . $applicationId, $fileName, 'public');

            SKPTDocument::create([
                'skpt_application_id' => $applicationId,
                'document_type' => 'additional',
                'document_name' => $validatedData['document_name'],
                'file_path' => $filePath,
                'file_size' => $file->getSize(),
                'file_extension' => $file->getClientOriginalExtension(),
                'is_required' => false,
                'is_additional' => true,
                'is_uploaded' => true,
                'verification_status' => 'PENDING',
                'verification_notes' => null,
                'verified_at' => null,
                'verified_by' => null,
            ]);

            DB::commit();

            return response()->json(['success' => 'Dokumen tambahan berhasil diunggah']);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'Terjadi kesalahan: ' . $e->getMessage()], 500);
        }
    }
    /**
     * Display the specified SKPT application.
     */
    public function show(SKPTApplication $skpt)
    {
        if (auth()->user()->hasRole('guest') && $skpt->created_by !== auth()->id()) {
            abort(403, 'Anda tidak memiliki akses ke data ini.');
        }
        $skpt->load([
            'documents' => function($query) {
                $query->orderBy('is_required', 'desc')->orderBy('document_type');
            },
            'additionalDocuments' => function($query) {
                $query->active()->orderBy('created_at', 'desc');
            }
        ]);

        $documentStats = $skpt->getDocumentStatistics();

        return view('admin.skpt.show', compact('skpt', 'documentStats'));
    }

    /**
     * Show the form for editing the specified SKPT application.
     */
    public function edit(SKPTApplication $skpt)
    {
        if (auth()->user()->hasRole('guest') && $skpt->created_by !== auth()->id()) {
            abort(403, 'Anda tidak memiliki akses ke data ini.');
        }
        $data_kampung = Kampung::orderBy('nama_kampung','ASC')->get();
        return view('admin.skpt.edit', compact('skpt', 'data_kampung'));
    }

    /**
     * Update the specified SKPT application.
     */
    public function update(Request $request, SKPTApplication $skpt)
    {
        $validatedData = $request->validate([
            // Data Pemohon
            'nama_pemohon' => 'required|string|max:255',
            'nik' => 'required|string|max:20|unique:skpt_applications,nik,' . $skpt->id,
            'no_ktp' => 'required|string|max:20',
            'tempat_lahir' => 'required|string|max:255',
            'tanggal_lahir' => 'required|date',
            'email' => 'nullable|email|max:255',
            'telepon' => 'nullable|string|max:15',
            'pekerjaan' => 'required|string|max:255',

            // Alamat Pemohon
            'alamat_rt' => 'required|string|max:3',
            'alamat_rw' => 'required|string|max:3',
            'alamat_kode_pos' => 'required|string|max:5',

            // Detail Tanah
            'letak_tanah' => 'required|string',
            'tanah_rt' => 'required|string|max:3',
            'tanah_rw' => 'required|string|max:3',
            'tanah_kode_pos' => 'required|string|max:5',
            'dusun' => 'required|string|max:255',
            'kelurahan_kampung' => 'required|string|max:255',
            'peruntukan' => 'required|in:Perkebunan,Pertanian,Perumahan,Industri,Lainnya',
            'luas' => 'required|numeric|min:0',
            'penanda_tanah' => 'required|in:Batas Pagar,Batas Alam,Batas Bangunan,Lainnya',
            'utara' => 'required|in:Tanah Kosong,Rumah,Kebun,Jalan,Sungai,Lainnya',
            'timur' => 'required|in:Tanah Kosong,Rumah,Kebun,Jalan,Sungai,Lainnya',
            'selatan' => 'required|in:Jalan Kampung,Rumah,Kebun,Tanah Kosong,Sungai,Lainnya',
            'barat' => 'required|in:Tanah Hasan,Rumah,Kebun,Tanah Kosong,Sungai,Lainnya',
        ]);

        try {
            DB::beginTransaction();
            $validatedData['updated_by'] = auth()->id();
            $skpt->update($validatedData);

            // Update data anak jika ada
            if ($request->has('anak')) {
                // Hapus data anak lama
                SkptAnak::where('skpt_id', $skpt->id)->delete();
                // Simpan data anak baru
                foreach ($request->anak as $anak) {
                    if (!empty($anak['nama_anak'])) {
                        SkptAnak::create([
                            'skpt_id'   => $skpt->id,
                            'nama_anak' => $anak['nama_anak'],
                            'usia'      => $anak['usia'] ?? null,
                        ]);
                    }
                }
            }

            // Simpan dokumen wajib/opsional jika ada perubahan
            $documents = [
                'ktp' => ['name' => 'Photo Copy KTP', 'is_required' => true, 'is_additional' => false],
                'kk' => ['name' => 'Photo Copy KK (Kartu Keluarga)', 'is_required' => true, 'is_additional' => false],
                'ktp_saksi' => ['name' => 'Photo Copy KTP Saksi Batas', 'is_required' => true, 'is_additional' => false],
                'surat_pengantar_rt' => ['name' => 'Surat Pengantar dari RT', 'is_required' => true, 'is_additional' => false],
                'surat_pengantar_kampung' => ['name' => 'Surat Pengantar dari Kepala Kampung', 'is_required' => true, 'is_additional' => false],
                'copy_surat_riwayat' => ['name' => 'Copy Surat Riwayat Peguasaan Fisik Bidang Tanah', 'is_required' => true, 'is_additional' => false],
                'surat_permohonan' => ['name' => 'Surat Permohonan SKPT', 'is_required' => true, 'is_additional' => false],
                'foto_bukti_1' => ['name' => 'Foto copy bukti yuridis penguasaan tanah negara hal-1', 'is_required' => true, 'is_additional' => false],
                'foto_bukti_2' => ['name' => 'Foto copy bukti yuridis penguasaan tanah negara hal-2', 'is_required' => true, 'is_additional' => false],
                'surat_keterangan' => ['name' => 'Surat Keterangan Kematian/Pembagian Waris', 'is_required' => false, 'is_additional' => true],
                'sppt_pbb' => ['name' => 'SPPT PBB', 'is_required' => false, 'is_additional' => true],
            ];
            foreach ($documents as $key => $doc) {
                if ($request->hasFile($key)) {
                    $file = $request->file($key);
                    $fileName = time() . '_' . $key . '.' . $file->getClientOriginalExtension();
                    $filePath = $file->storeAs('documents/skpt/' . $skpt->id, $fileName, 'public');
                    SKPTDocument::updateOrCreate(
                        [
                            'skpt_application_id' => $skpt->id,
                            'document_type' => $key
                        ],
                        [
                            'document_name' => $doc['name'],
                            'file_path' => $filePath,
                            'file_size' => $file->getSize(),
                            'file_extension' => $file->getClientOriginalExtension(),
                            'is_required' => $doc['is_required'],
                            'is_additional' => $doc['is_additional'],
                            'is_uploaded' => true,
                            'verification_status' => 'PENDING',
                            'verification_notes' => null,
                            'verified_at' => null,
                            'verified_by' => null,
                        ]
                    );
                }
            }

            // Simpan dokumen tambahan jika ada
            if ($request->hasFile('additional_documents')) {
                foreach ($request->file('additional_documents') as $index => $file) {
                    if ($file) {
                        $fileName = time() . '_additional_' . $index . '.' . $file->getClientOriginalExtension();
                        $filePath = $file->storeAs('documents/skpt/' . $skpt->id, $fileName, 'public');
                        $description = $request->input('additional_descriptions')[$index] ?? '';
                        SKPTDocument::create([
                            'skpt_application_id' => $skpt->id,
                            'document_type' => 'additional_' . $index,
                            'document_name' => 'Dokumen Tambahan ' . ($index + 1),
                            'description' => $description,
                            'file_path' => $filePath,
                            'file_size' => $file->getSize(),
                            'file_extension' => $file->getClientOriginalExtension(),
                            'is_required' => false,
                            'is_additional' => true,
                            'is_uploaded' => true,
                            'verification_status' => 'PENDING',
                            'verification_notes' => null,
                            'verified_at' => null,
                            'verified_by' => null,
                        ]);
                    }
                }
            }

            DB::commit();
            return redirect()->route('skpt.show', $skpt)
                           ->with('success', 'Data permohonan SKPT berhasil diperbarui.');

        } catch (\Exception $e) {
            return redirect()->back()
                           ->withInput()
                           ->with('error', 'Terjadi kesalahan saat memperbarui data: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified SKPT application.
     */
    public function destroy(SKPTApplication $skpt)
    {
        if (auth()->user()->hasRole('guest') && $skpt->created_by !== auth()->id()) {
            abort(403, 'Anda tidak memiliki akses ke data ini.');
        }
        try {
            DB::beginTransaction();

            // Hapus semua file dokumen terkait
            foreach ($skpt->documents as $document) {
                if ($document->fileExists()) {
                    $document->deleteFile();
                }
            }

            foreach ($skpt->additionalDocuments as $document) {
                if ($document->fileExists()) {
                    $document->deleteFile();
                }
            }

            // Hapus aplikasi (cascade akan hapus dokumen dari database)
            $noRegistrasi = $skpt->no_registrasi;
            $skpt->delete();

            DB::commit();

            return redirect()->route('skpt.index')
                           ->with('success', 'Permohonan SKPT No. ' . $noRegistrasi . ' berhasil dihapus.');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                           ->with('error', 'Terjadi kesalahan saat menghapus data: ' . $e->getMessage());
        }
    }

    /**
     * Upload document for SKPT application.
     */
    public function uploadDocument(Request $request, SKPTApplication $skpt, SKPTDocument $document)
    {
        $request->validate([
            'file' => 'required|file|mimes:' . implode(',', SKPTDocument::getAllowedExtensions()) . '|max:' . (SKPTDocument::getMaxFileSize() / 1024),
        ], [
            'file.mimes' => 'File harus berformat: ' . implode(', ', SKPTDocument::getAllowedExtensions()),
            'file.max' => 'Ukuran file maksimal ' . (SKPTDocument::getMaxFileSize() / 1024 / 1024) . 'MB',
        ]);

        try {
            $file = $request->file('file');
            $fileName = time() . '_' . $file->getClientOriginalName();
            $filePath = $file->storeAs('skpt_documents/' . $skpt->id, $fileName, 'public');

            // Hapus file lama jika ada
            if ($document->fileExists()) {
                $document->deleteFile();
            }

            // Update document record
            $document->update([
                'file_path' => $filePath,
                'file_size' => $file->getSize(),
                'file_extension' => $file->getClientOriginalExtension(),
                'is_uploaded' => true,
                'verification_status' => 'PENDING'
            ]);

            // Update status kelengkapan dokumen aplikasi
            $skpt->updateDocumentStatus();

            return redirect()->back()
                           ->with('success', 'Dokumen ' . $document->document_name . ' berhasil diupload.');

        } catch (\Exception $e) {
            return redirect()->back()
                           ->with('error', 'Terjadi kesalahan saat mengupload dokumen: ' . $e->getMessage());
        }
    }

    /**
     * Upload additional document.
     */
    public function uploadAdditionalDocument(Request $request, SKPTApplication $skpt)
    {
        $request->validate([
            'document_name' => 'required|string|max:255',
            'document_description' => 'nullable|string|max:1000',
            'file' => 'required|file|mimes:' . implode(',', SKPTAdditionalDocument::getAllowedExtensions()) . '|max:' . (SKPTAdditionalDocument::getMaxFileSize() / 1024),
        ], [
            'file.mimes' => 'File harus berformat: ' . implode(', ', SKPTAdditionalDocument::getAllowedExtensions()),
            'file.max' => 'Ukuran file maksimal ' . (SKPTAdditionalDocument::getMaxFileSize() / 1024 / 1024) . 'MB',
        ]);

        try {
            $file = $request->file('file');
            $fileName = time() . '_' . $file->getClientOriginalName();
            $filePath = $file->storeAs('skpt_additional_documents/' . $skpt->id, $fileName, 'public');

            // Create additional document record
            SKPTAdditionalDocument::create([
                'skpt_application_id' => $skpt->id,
                'document_name' => $request->document_name,
                'document_description' => $request->document_description,
                'file_path' => $filePath,
                'original_filename' => $file->getClientOriginalName(),
                'file_size' => $file->getSize(),
                'file_extension' => $file->getClientOriginalExtension(),
                'verification_status' => 'PENDING'
            ]);

            return redirect()->back()
                           ->with('success', 'Dokumen tambahan berhasil diupload.');

        } catch (\Exception $e) {
            return redirect()->back()
                           ->with('error', 'Terjadi kesalahan saat mengupload dokumen tambahan: ' . $e->getMessage());
        }
    }

    /**
     * Verify document.
     */
    public function verifyDocument(Request $request, SKPTApplication $skpt, SKPTDocument $document)
    {
        $request->validate([
            'action' => 'required|in:approve,reject',
            'notes' => 'nullable|string|max:1000'
        ]);

        try {
            if ($request->action === 'approve') {
                $document->verify(auth()->id(), $request->notes);
                $message = 'Dokumen berhasil diverifikasi.';
            } else {
                $document->reject(auth()->id(), $request->notes);
                $message = 'Dokumen berhasil ditolak.';
            }

            // Update status aplikasi berdasarkan verifikasi dokumen
            // $skpt->updateApplicationStatus();
            $skpt->updateDocumentStatus();
            return redirect()->back()->with('success', $message);

        } catch (\Exception $e) {
            return redirect()->back()
                           ->with('error', 'Terjadi kesalahan saat memverifikasi dokumen: ' . $e->getMessage());
        }
    }

    /**
     * Verify additional document.
     */
   public function verifyAdditionalDocument(Request $request, SKPTApplication $skpt, SKPTAdditionalDocument $document)
    {
        $request->validate([
            'action' => 'required|in:approve,reject',
            'notes' => 'nullable|string|max:1000'
        ]);

        try {
            if ($request->action === 'approve') {
                $document->verify(auth()->id(), $request->notes);
                $message = 'Dokumen tambahan berhasil diverifikasi.';
            } else {
                $document->reject(auth()->id(), $request->notes);
                $message = 'Dokumen tambahan berhasil ditolak.';
            }

            return redirect()->back()->with('success', $message);

        } catch (\Exception $e) {
            return redirect()->back()
                        ->with('error', 'Terjadi kesalahan saat memverifikasi dokumen tambahan: ' . $e->getMessage());
        }
    }

    /**
     * Download document.
     */
    public function downloadDocument(SKPTDocument $document)
    {
        if (!$document->fileExists()) {
            return redirect()->back()->with('error', 'File tidak ditemukan.');
        }

        return Storage::disk('public')->download($document->file_path, $document->document_name . '.' . $document->file_extension);
    }

    /**
     * Download additional document.
     */
    public function downloadAdditionalDocument(SKPTAdditionalDocument $document)
    {
        if (!$document->fileExists()) {
            return redirect()->back()->with('error', 'File tidak ditemukan.');
        }

        return Storage::disk('public')->download($document->file_path, $document->original_filename);
    }

    /**
     * Delete additional document.
     */
    public function deleteAdditionalDocument(SKPTApplication $skpt, SKPTAdditionalDocument $document)
    {
        try {
            $document->deleteFile();
            $document->delete();

            return redirect()->back()->with('success', 'Dokumen tambahan berhasil dihapus.');

        } catch (\Exception $e) {
            return redirect()->back()
                           ->with('error', 'Terjadi kesalahan saat menghapus dokumen: ' . $e->getMessage());
        }
    }

    /**
     * Get dashboard statistics.
     */
    public function dashboard()
    {
        $stats = [
            'total_applications' => SKPTApplication::count(),
            'completed_applications' => SKPTApplication::where('status_permohonan', 'SELESAI')->count(),
            'pending_applications' => SKPTApplication::where('status_permohonan', 'BELUM SELESAI')->count(),
            'complete_documents' => SKPTApplication::where('status_kelengkapan_dokumen', 'LENGKAP')->count(),
            'pending_verifications' => SKPTDocument::where('verification_status', 'PENDING')->where('is_uploaded', true)->count(),
            'recent_applications' => SKPTApplication::with('documents')->orderBy('created_at', 'desc')->take(5)->get(),
        ];

        return view('admin.skpt.dashboard', compact('stats'));
    }

   public function updateStatus(Request $request, $id)
    {
        $request->validate(['status' => 'required|in:SELESAI,BELUM SELESAI']);

    DB::beginTransaction();
        try {
            $application = SKPTApplication::findOrFail($id);
            $application->status_permohonan = $request->status;
            $application->save();

            // Cek jika sudah ada peninjauan untuk SKPT ini
            $existingPeninjauan = Peninjauan::where('skpt_id', $application->id)->first();
            if ($existingPeninjauan) {
                // Jika sudah ada peninjauan, pastikan tidak membuat ulang BeritaAcaraPeninjauan jika sudah ada
                $existingBA = BeritaAcaraPeninjauan::where('peninjauan_id', $existingPeninjauan->id)->count();
                // if ($existingBA >= 3) {
                //     return response()->json([
                //         'success' => false,
                //         'message' => 'Peninjauan dan Berita Acara sudah ada untuk permohonan ini.'
                //     ]);
                // }
                // Jika peninjauan ada tapi BA kurang dari 3, tambahkan yang kurang
                $types = ['pengukuran', 'persetujuan', 'penelitian'];
                $existingTypes = BeritaAcaraPeninjauan::where('peninjauan_id', $existingPeninjauan->id)
                    ->pluck('type')->toArray();
                foreach ($types as $type) {
                    if (!in_array($type, $existingTypes)) {
                        $prefix = '';
                        if ($type === 'pengukuran') {
                            $prefix = 'BA-PBT-PENGUKURAN-';
                        } elseif ($type === 'persetujuan') {
                            $prefix = 'BA-PPT-PERSETUJUAN-';
                        } elseif ($type === 'penelitian') {
                            $prefix = 'BA-PPUT-PENELITIAN-';
                        }
                        $nomorBeritaAcara = str_replace('BA-', $prefix, $existingPeninjauan->nomor_ba_kesaksian);
                        BeritaAcaraPeninjauan::create([
                            'peninjauan_id'      => $existingPeninjauan->id,
                            'nomor_berita_acara' => $nomorBeritaAcara,
                            'type'               => $type,
                            'status'             => 'tidak sesuai',
                            'status_verifikasi'  => 'tidak sesuai',

                            'verifikator_id'     => null,
                        ]);
                    }
                }
                DB::commit();
                return response()->json([
                    'success' => true,
                    'message' => 'Berita Acara yang kurang telah ditambahkan.'
                ]);
            }

            // Jika belum ada peninjauan, buat baru
            $peninjauan = Peninjauan::create([
                'skpt_id'    => $id,
                'status'     => 'draft',
                'created_by' => auth()->id(),
                'updated_by' => auth()->id(),
                'nomor_ba_kesaksian' => $this->generateNomorBA(),
            ]);

            $types = ['pengukuran', 'persetujuan', 'penelitian'];
            foreach ($types as $type) {
                $prefix = '';
                if ($type === 'pengukuran') {
                    $prefix = 'BA-PBT-PENGUKURAN-';
                } elseif ($type === 'persetujuan') {
                    $prefix = 'BA-PPT-PERSETUJUAN-';
                } elseif ($type === 'penelitian') {
                    $prefix = 'BA-PPUT-PENELITIAN-';
                }
                $nomorBeritaAcara = str_replace('BA-', $prefix, $peninjauan->nomor_ba_kesaksian);
                BeritaAcaraPeninjauan::create([
                    'peninjauan_id'      => $peninjauan->id,
                    'nomor_berita_acara' => $nomorBeritaAcara,
                    'type'               => $type,
                    'status'             => 'tidak sesuai',
                    'status_verifikasi'  => 'tidak sesuai',

                    'verifikator_id'     => null,
                ]);
            }
            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Status berhasil diperbarui dan peninjauan baru telah dibuat.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

     private function generateNomorBA()
    {
        $prefix = 'BA-' . date('Ymd'); // contoh: BA-20250921

        // Cari nomor terakhir di hari ini
        $lastRecord = Peninjauan::whereDate('created_at', now())
            ->where('nomor_ba_kesaksian', 'like', $prefix . '%')
            ->latest('id')
            ->first();

        if ($lastRecord) {
            // Ambil urutan terakhir
            $lastNumber = (int) substr($lastRecord->nomor_ba_kesaksian, -4);
            $newNumber = str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
        } else {
            $newNumber = '0001';
        }

        return $prefix . '-' . $newNumber;
    }

     public function getTableData(Request $request)
    {
        // dd('tes');
        try {
            $user = auth()->user();
            $query = SKPTApplication::with(['dataPeninjauan', 'dataPeninjauan.dataKecamatan']);
            // Jika guest, filter hanya data miliknya
            if (auth()->user()->hasRole('guest')) {
                $query->where('created_by', $user->id);
            }
            // Apply search filter
            if ($request->search) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('nama_pemohon', 'LIKE', "%{$search}%")
                      ->orWhere('no_registrasi', 'LIKE', "%{$search}%")
                      ->orWhere('nik', 'LIKE', "%{$search}%");
                });
            }

            // Apply status filter
            if ($request->status && $request->status !== 'all') {
                $query->where('status_permohonan', $request->status);
            }

            // Order by latest first
            $query->orderBy('created_at', 'DESC');

            // Paginate
            $perPage = $request->per_page ?? 20;
            $page = $request->page ?? 1;
            $offset = ($page - 1) * $perPage;

            $data = $query->skip($offset)
                         ->take($perPage)
                         ->get();

            // Transform data for frontend
            $transformedData = $data->map(function($item) {
                return [
                    'id' => $item->id,
                    'no_registrasi' => $item->no_registrasi,
                    'nama_pemohon' => $item->nama_pemohon,
                    'nik' => $item->nik,
                    'luas' => $item->luas,
                    'status_permohonan' => $item->status_permohonan,
                    'created_at' => $item->created_at,
                    'jalan_gg_blok' => $item->dataPeninjauan->jalan_gg_blok ?? null,
                    'rt' => $item->dataPeninjauan->rt ?? null,
                    'rw' => $item->dataPeninjauan->rw ?? null,
                    'kecamatan' => $item->dataPeninjauan->dataKecamatan->nama_kecamatan ?? null,
                    'has_coordinates' => $item->dataPeninjauan && $item->dataPeninjauan->coordinates->count() > 0
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedData
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get total count of data
     */
    public function getTotalCount(Request $request)
    {
        try {
            $user = auth()->user();
            $query = SKPTApplication::query();
            if (auth()->user()->hasRole('guest')) {
                $query->where('created_by', $user->id);
            }
            // Apply search filter
            if ($request->search) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('nama_pemohon', 'LIKE', "%{$search}%")
                      ->orWhere('no_registrasi', 'LIKE', "%{$search}%")
                      ->orWhere('nik', 'LIKE', "%{$search}%");
                });
            }

            // Apply status filter
            if ($request->status && $request->status !== 'all') {
                $query->where('status_permohonan', $request->status);
            }

            $total = $query->count();

            return response()->json(['total' => $total]);

        } catch (\Exception $e) {
            return response()->json(['total' => 0], 500);
        }
    }

    /**
     * Get polygon data for a specific SKPT application
     */
    public function getPolygonData(Request $request)
    {
        try {
            $id = $request->id;

            $skpt = SKPTApplication::with([
                'dataPeninjauan.coordinates',
                'dataPeninjauan.dataKecamatan'
            ])->find($id);

            if (!$skpt) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data not found'
                ], 404);
            }

            // Check if has coordinates
            if (!$skpt->dataPeninjauan || $skpt->dataPeninjauan->coordinates->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'No coordinates available'
                ]);
            }

            // Prepare coordinates
            $coordinates = $skpt->dataPeninjauan->coordinates->map(function($coord) {
                return [
                    'lat' => (float) $coord->latitude,
                    'lng' => (float) $coord->longitude
                ];
            });

            // Prepare response data
            $data = [
                'success' => true,
                'id' => $skpt->id,
                'no_registrasi' => $skpt->no_registrasi,
                'nama_pemohon' => $skpt->nama_pemohon,
                'nik' => $skpt->nik,
                'luas' => $skpt->luas,
                'status' => $skpt->status_permohonan,
                'created_at' => $skpt->created_at,
                'alamat' => $skpt->dataPeninjauan->jalan_gg_blok ?? '',
                'kecamatan' => $skpt->dataPeninjauan->dataKecamatan->nama_kecamatan ?? '',
                'coordinates' => $coordinates
            ];

            return response()->json($data);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading polygon data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get detail data for modal
     */
    public function getDetailData(Request $request)
    {
        try {
            $id = $request->id;

            $skpt = SKPTApplication::with([
                'dataPeninjauan',
                'dataPeninjauan.dataKecamatan',
                'dataPeninjauan.dataKelurahan'
            ])->find($id);

            if (!$skpt) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data not found'
                ], 404);
            }

            // Prepare detailed response
            $data = [
                'success' => true,
                'pemohon' => [
                    'nama' => $skpt->nama_pemohon,
                    'nik' => $skpt->nik,
                    'no_hp' => $skpt->no_hp,
                    'alamat' => $skpt->alamat_pemohon,
                    'no_registrasi' => $skpt->no_registrasi
                ],
                'tanah' => [
                    'alamat' => $skpt->dataPeninjauan->jalan_gg_blok ?? '-',
                    'rt' => $skpt->dataPeninjauan->rt ?? '-',
                    'rw' => $skpt->dataPeninjauan->rw ?? '-',
                    'kecamatan' => $skpt->dataPeninjauan->dataKecamatan->nama_kecamatan ?? '-',
                    'kelurahan' => $skpt->dataPeninjauan->dataKelurahan->nama_kelurahan ?? '-',
                    'luas' => $skpt->luas,
                    'batas_utara' => $skpt->dataPeninjauan->batas_utara ?? '-',
                    'batas_selatan' => $skpt->dataPeninjauan->batas_selatan ?? '-',
                    'batas_timur' => $skpt->dataPeninjauan->batas_timur ?? '-',
                    'batas_barat' => $skpt->dataPeninjauan->batas_barat ?? '-'
                ],
                'status' => [
                    'permohonan' => $skpt->status_permohonan,
                    'dokumen' => $skpt->status_kelengkapan_dokumen,
                    'tanggal_permohonan' => $skpt->created_at->format('d F Y'),
                    'tanggal_update' => $skpt->updated_at->format('d F Y H:i')
                ]
            ];

            return response()->json($data);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading detail: ' . $e->getMessage()
            ], 500);
        }
    }

      public function filterSkpt(Request $request)
    {
        try {
            $user = auth()->user();
            $query = SKPTApplication::with(['dataPeninjauan.coordinates', 'dataPeninjauan.dataKecamatan']);
            if (auth()->user()->hasRole('guest')) {
                $query->where('created_by', $user->id);
            }
            // Apply search filter
            if ($request->has('search') && !empty($request->search)) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('nama_pemohon', 'LIKE', "%{$search}%")
                      ->orWhere('no_registrasi', 'LIKE', "%{$search}%")
                      ->orWhere('nik', 'LIKE', "%{$search}%")
                      ->orWhereHas('dataPeninjauan', function($q2) use ($search) {
                          $q2->where('jalan_gg_blok', 'LIKE', "%{$search}%");
                      });
                });
            }

            // Apply status filter
            if ($request->has('status') && $request->status !== 'all') {
                $query->where('status_permohonan', $request->status);
            }

            // Apply date filters
            if ($request->has('date_from') && !empty($request->date_from)) {
                $query->whereDate('created_at', '>=', $request->date_from);
            }

            if ($request->has('date_to') && !empty($request->date_to)) {
                $query->whereDate('created_at', '<=', $request->date_to);
            }

            // Only get records with coordinates for map
            $query->whereHas('dataPeninjauan.coordinates');

            $skptApplications = $query->get();

            // Prepare map data
            $mapData = [];
            foreach ($skptApplications as $skpt) {
                if ($skpt->dataPeninjauan && $skpt->dataPeninjauan->coordinates->count() > 0) {
                    $coordinates = $skpt->dataPeninjauan->coordinates->map(function($coord) {
                        return [
                            'lat' => (float) $coord->latitude,
                            'lng' => (float) $coord->longitude
                        ];
                    })->toArray();

                    $mapData[] = [
                        'id' => $skpt->id,
                        'no_registrasi' => $skpt->no_registrasi,
                        'nama_pemohon' => $skpt->nama_pemohon,
                        'status' => $skpt->status_permohonan,
                        'luas' => $skpt->luas,
                        'alamat' => $skpt->dataPeninjauan->jalan_gg_blok ?? '',
                        'kecamatan' => $skpt->dataPeninjauan->dataKecamatan->nama_kecamatan ?? '',
                        'coordinates' => $coordinates,
                        'color' => $skpt->status_permohonan == 'SELESAI' ? '#28a745' : '#ffc107'
                    ];
                }
            }

            return response()->json([
                'success' => true,
                'data' => $mapData,
                'count' => count($mapData)
            ]);

        } catch (\Exception $e) {
            Log::error('Filter error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error filtering data'
            ], 500);
        }
    }




}
